<?php namespace App;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Cviebrock\EloquentSluggable\Sluggable;
use Illuminate\Support\Str;

use Request;

class Article extends Model
{
    use Sluggable;

    protected $fillable
        = [
            'comment_system',
            'title',
            'sub_title',
            'meta_title',
            'meta_description',
            'meta_keywords',
            'content',
            'published_at',
            'slug',
            'newsletter',
            'top_article',
            'article_author_id',
            'article_category_id',
            'article_type_id',
            'article_magazine_edition_id',
            'job_sector',
            'job_manages_others',
            'job_application_closing_date',
            'job_experience_required',
            'job_employment_type',
            'job_work_schedule',
            'job_link',
            'views',
            'views_month',
            'views_week',
        ];

    protected $with = [
        'articleImages',
    ];

    public function toSearchableArray()
    {

        return [
            'comment_system'              => $this->comment_system,
            'title'                       => $this->title,
            'sub_title'                   => $this->sub_title,
            'meta_title'                  => $this->meta_title,
            'meta_description'            => $this->meta_description,
            'meta_keywords'               => $this->meta_keywords,
            'content'                     => $this->algolia_content,
            'published_at'                => $this->published_at,
            'slug'                        => $this->slug,
            'newsletter'                  => $this->newsletter,
            'top_article'                 => $this->top_article,
            'article_author_id'           => $this->article_author_id,
            'article_category_id'         => $this->article_category_id,
            'article_type_id'             => $this->article_type_id,
            'article_magazine_edition_id' => $this->article_magazine_edition_id,
        ];
    }

    public function getRouteKeyName()
    {

        return 'slug';
    }

    /**
     * Return the sluggable configuration array for this model.
     *
     * @return array
     */
    public function sluggable()
    {

        return [
            'slug' => [
                'source' => 'title',
            ],
        ];
    }

    /**
     *
     *
     *
     *
     *
     * RELATIONSHIPS
     *
     *
     *
     *
     *
     */

    /**
     *
     *
     * belongsTo Relationships - articleAuthor, articleCategory, articleType, articleMagazineEdition
     *
     *
     */

    //Author
    public function articleAuthor()
    {

        return $this->belongsTo('App\ArticleAuthor');
    }

    //Category
    public function articleCategory()
    {

        return $this->belongsTo('App\ArticleCategory')
            ->withDefault([
                'title' => "(None)"
            ]);
    }

    //Type
    public function articleType()
    {

        return $this->belongsTo('App\ArticleType');
    }

    //Magazine Edition
    public function articleMagazineEdition()
    {

        return $this->belongsTo('App\ArticleMagazineEdition');
    }

    /**
     *
     *
     * belongsToMany Relationships - articleImages, newsletters
     *
     *
     */

    //Images
    public function articleImages()
    {

        return $this->belongsToMany('App\ArticleImage');
    }

    //Newsletters
    public function newsletters()
    {

        return $this->belongsToMany('App\Newsletter');
    }

    //Title Tags
    public function articleTitleTags()
    {

        return $this->belongsToMany('App\ArticleTitleTag');
    }

    //Users
    public function user()
    {

        return $this->belongsToMany('App\User');
    }

    /**
     *
     *
     * hasMany Relationships - user, articleTitleTags, articleSlugRedirects
     *
     *
     */

    //Slug Redirects
    public function articleSlugRedirect()
    {

        return $this->hasMany('App\ArticleSlugRedirect');
    }

    /**
     *
     *
     *
     *
     *
     * SCOPES
     *
     *
     *
     *
     *
     */

    public function scopeRandomArticles($query)
    {

        return ArticleType::inRandomOrder()->whereNotIn('id', ['7', '1'])->first()->articles()->published()->limit(6)->get();
    }

    public function scopePublished($query)
    {

        $query->where('published_at', '!=', '0000-00-00 00:00:00')->where('published_at', '<=', Carbon::now())->orderBy('published_at', 'DESC');
    }

    public function scopeTopArticle($query)
    {

        return $query->whereTopArticle('1')->first();
    }

    public function scopeFeaturedArticles($query)
    {

        return $query->whereArticleTypeId('3');
    }

    public function scopeNewsletterCorporateNews($query, $official_date = null)
    {

        $official_date = $official_date ? Carbon::parse($official_date)->subDay(1)->toDateString() : Carbon::now()->toDateString();

        $query->where([['newsletter', '1'], ['published_at', 'LIKE', '%' . $official_date . '%']]);
    }

    /**
     *
     *
     *
     *
     *
     * GETTERS
     *
     *
     *
     *
     *
     */

    /**
     * Returns the published at as a normal m/d/Y instead of the standard server datetime format
     *
     * @param $published_at
     *
     * @return string
     */
    public function getPublishedAtArticleAttribute($published_at)
    {

        return empty($published_at) || $published_at == '0000-00-00 00:00:00' ? $published_at : Carbon::parse($published_at)->format('m/d/Y h:i:s');
    }

    /**
     * Access the article image
     *
     * @return string
     */
    public function getArticleImageAttribute()
    {

        return $this->articleImages->where('articleImageType.name', 'article')->first();
    }

    /**
     * Access the newsletter primary image
     *
     * @return string
     */
    public function getNewsletterPrimaryImageAttribute()
    {

        return $this->articleImages->where('articleImageType.name', 'newsletter_primary_image')->first();
    }

    /**
     * Access the newsletter secondary image
     *
     * @return string
     */
    public function getNewsletterSecondaryImageAttribute()
    {

        return $this->articleImages->where('articleImageType.name', 'newsletter_secondary_image')->first();
    }

    /**
     * Access the script image
     *
     * @return string
     */
    public function getScriptImageAttribute()
    {

        return $this->articleImages->where('articleImageType.name', 'script')->first();
    }

    /**
     * Used by the Newsletter to create a link for the article
     *
     * @return string
     */
    public function getLinkAttribute()
    {

        return $this->articleType ? Request::root() . '/' . $this->articleType->slug . '/' . $this->slug : '#';
    }

    //Article Type
    public function getArticleTypeAttribute()
    {

        return $this->articleType()->first();
    }

    /**
     * Content Summary
     *
     * @return string
     */
    public function getContentSummaryAttribute()
    {
        return strip_tags(Str::words($this->content, 80, '...'));
    }

    // Content substring
    public function getContentSubstringAttribute()
    {
        return strip_tags(Str::substr($this->content, 0, 300)) . '...';
    }

    // Content substring short
    public function getContentSubshortAttribute()
    {
        return strip_tags(Str::substr($this->content, 0, 200)) . '...';
    }

    public function getAlgoliaContentAttribute()
    {
        //size to keep all records under (in KB)
        $record_size_limit = 18000;

        //php word limit
        $php_word_limit = 3000;

        //get the size of all the other records to add into the total we are trying to stay under (18KB)
        $total_record_size = 0;

        foreach ($this->attributes as $key => $item) {
            if ($key != 'content') {
                $total_record_size += strlen($item);
            }
        };

//        dump('ID: ' . $this->id, 'before size: ' . strlen($this->content), 'before count: ' . str_word_count($this->content), 'before $total_record_size: ' . $total_record_size);

        //check if the entire record is under 18KB
        if ((strlen($this->content) + $total_record_size) >= $record_size_limit) {
            //get the current word count of content
            $word_count = str_word_count($this->content);

            //this check is for php's character processing limit
            $word_count = $word_count > $php_word_limit ? $php_word_limit : $word_count;

            //remove 100 words at a time until the entire record is under 18KB
            do {
                $this->content = Str::words($this->content, $word_count);

                $word_count -= 100;
            } while ((strlen($this->content) + $total_record_size) > $record_size_limit);

//            dd('after size: ' . strlen($this->content), 'after count: ' . str_word_count($this->content));
        }

        return $this->content;
    }

    /**
     *
     *
     *
     *
     *
     * SETTERS
     *
     *
     *
     *
     *
     */

    /**
     * @param $published_at
     */
    public function setPublishedAtAttribute($published_at)
    {

        $this->attributes['published_at'] = empty($published_at) || $published_at == '0000-00-00 00:00:00' ? $published_at : Carbon::parse($published_at)->toDateTimeString();
    }

    /**
     * @param $article_type_id
     */
    public function setArticleTypeIdAttribute($article_type_id)
    {

        $this->attributes['article_type_id'] = $article_type_id ?: null;
    }

    public function setArticleMagazineEditionIdAttribute($article_magazine_edition_id)
    {

        $this->attributes['article_magazine_edition_id'] = $article_magazine_edition_id ?: null;
    }

    public function setArticleAuthorIdAttribute($article_author_id)
    {

        $this->attributes['article_author_id'] = $article_author_id ?: null;
    }

    public function setMetaDescriptionAttribute()
    {

        $this->attributes['meta_description'] = str_limit(preg_replace('/(<.*?>)|(&.*?;)/', '', $this->attributes['content']), $limit = 100, $end = '...');
    }

    public function setJobLinkAttribute($job_link)
    {

            // $this->attributes['job_link'] = strpos($job_link, 'http://') === FALSE && strpos($job_link, 'https://') === FALSE ? 'http://' . $job_link : $job_link;
            $this->attributes['job_link'] = $job_link;
    }
}
