<?php

namespace App\Http\Controllers;

use App\Article;
use App\ArticleType;
use App\Repositories\ArticleTypesRepository;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Redirect;

class ArticleTypesController extends Controller
{

    protected $articleTypesRepository;

    public function __construct(ArticleTypesRepository $articleTypesRepository)
    {

        $this->middleware('auth', [
            'except' => [
                'corporateArticleList',
                'podcastArticleList',
                'jobArticleList',
                'moreArticles',
                'jobArticleList_redirected',
                'moreJobArticles',
            ],
        ]);

        $this->articleTypesRepository = $articleTypesRepository;
    }

    /**
     *
     *
     *
     * CUSTOM ROUTE METHODS
     *
     *
     *
     */

    /**
     * Page shows lists of type article categories
     *
     * @param $category
     */
    public function corporateArticleList()
    {

        $page_items['category']             = ArticleType::whereSlug('corporate')->first();
        $page_items['main_articles']        = $page_items['category']->articles()->published()->limit(10)->get();
        $page_items['side_article_content'] = Article::randomArticles();

        return view('site.article_list', $page_items);
    }

    /**
     * Page shows lists of type article categories
     *
     * @param $category
     */
    public function podcastArticleList()
    {

        $page_items['category']             = ArticleType::whereSlug('podcast')->first();
        $page_items['main_articles']        = $page_items['category']->articles()->published()->paginate(15);
        $page_items['side_article_content'] = Article::randomArticles();

        /* $page_items['top_stories'] = Article::published()
            ->where('top_article', '!=', '1')
            ->whereNotIn('article_type_id', ['2', '7', '9'])
            ->limit(4)
            ->get(); */
            
        return view('site.podcast_article_list', $page_items);
    }

    /**
     * Page shows lists of type article categories
     *
     * @param $category
     */
    public function jobArticleList()
    {

        $page_items['category']             = ArticleType::whereSlug('job')->first();
        $page_items['main_articles']        = $page_items['category']->articles()->published()->where('published_at', '>=', Carbon::now()->subMonth()->toDateTimeString())->limit(10)->get();
        $page_items['side_article_content'] = Article::randomArticles();

        /* $page_items['top_stories'] = Article::published()
            ->where('top_article', '!=', '1')
            ->whereNotIn('article_type_id', ['2', '7', '9'])
            ->limit(4)
            ->get(); */

        return view('site.job_article_list', $page_items);
    }

    public function jobArticleList_redirected()
    {
        Redirect::to('/jobs', 301)->send();
    }

    /**
     *
     *
     *
     * RESTFUL METHODS
     *
     *
     *
     */

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index(Request $request)
    {

        //if a query exists from Scout get the string
        $query = $request->input('query');

        //check if the query is empty if it isn't use the scout search method before paginating
        $collection = empty($query) ? ArticleType::paginate(10) : ArticleType::search($query)->paginate(10);

        return view('admin.partials.index_page.index', $this->articleTypesRepository->indexElements(compact('collection')));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        return view($this->articleTypesRepository->getAdminFolderPath() . '.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        ArticleType::create($request->all());

        return redirect('article/type');
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(ArticleType $type)
    {

        return view($this->articleTypesRepository->getAdminFolderPath() . '.edit', compact('type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ArticleType $type)
    {

        $type->update($request->all());

        return redirect('article/type');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(ArticleType $type)
    {

        $type->delete();

        return redirect('article/type');
    }

    /**
     *
     *
     *
     * AJAX
     *
     *
     *
     */

    public function moreArticles(Request $request)
    {

        $limit  = $request->limit;
        $offset = $request->offset;
        $slug   = $request->slug;

        $articles = ArticleType::whereSlug($slug)->first()->articles()->published()->offset($offset)->limit($limit)->get();

        return view('site.partials.article.more_articles_content', compact('articles'));
    }

    public function moreJobArticles(Request $request)
    {

        $limit  = $request->limit;
        $offset = $request->offset;

        $articles = ArticleType::whereSlug('job')->first()->articles()->published()->where('published_at', '>=', Carbon::now()->subMonth()->toDateTimeString())->offset($offset)->limit($limit)->get();

        return view('site.partials.article.more_articles_content', compact('articles'));
    }
}
