<?php

namespace App\Http\Controllers;

use Grids;
use Nayjest\Grids\Components\FiltersRow;
use Nayjest\Grids\Components\Footer;
use Nayjest\Grids\Components\Header;
use Nayjest\Grids\Components\Pager;
use Nayjest\Grids\Components\RenderFunc;
use Nayjest\Grids\DataProvider;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\EloquentDataRow;
use Nayjest\Grids\FieldConfig;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\SelectFilterConfig;
use Nayjest\Grids\Grid;
use Nayjest\Grids\GridConfig;
use Nayjest\Grids\IdFieldConfig;
use Nayjest\Grids\ObjectDataRow;
use Nayjest\Grids\Components\ShowingRecords;
use Nayjest\Grids\Components\TFoot;
use Nayjest\Grids\Components\THead;
use Nayjest\Grids\Components\HtmlTag;
use Nayjest\Grids\Components\ColumnHeadersRow;
use Nayjest\Grids\Components\OneCellRow;
use Nayjest\Grids\Components\Base\RenderableRegistry;

use App\Banner;
use App\BannerType;
use App\BannerLog;
use App\Http\Requests\BannerRequest;
use App\Jobs\CreateBannerReports;
use App\Marex\Reports\BannerReport;
use App\Newsletter;
use App\Repositories\banner_repository;
use App\Repositories\BannersRepository;
use Auth;
use Illuminate\Http\Request;
use Log;
use Storage;
use Mail;

class BannersController extends Controller
{


    /**
     * @var BannerReport
     */

    private $banner_report;
    /**
     * @var banner_repository
     */
    private $banner_repository;

    /**
     * BannersController constructor.
     * @param BannersRepository $banner_repository
     * @param BannerReport $banner_report
     */
    public function __construct(BannersRepository $banner_repository, BannerReport $banner_report)
    {

        $this->middleware('auth');

        $this->banner_report = $banner_report;
        $this->banner_repository = $banner_repository;
    }

    /**
     *
     *
     *
     * CUSTOM ROUTE METHODS
     *
     *
     *
     */

    public function reports()
    {
        $result = \DB::table('banners')
                    ->groupBy('company')
                    ->pluck('company');

        $companies = json_encode($result);

        return view('admin.banners.reports', compact('companies'));
    }

    public function report($banner_id)
    {

        $banner = Banner::findOrFail($banner_id);

        $banner_content = $this->banner_report->report($banner);

        if ($banner_content) {
            $banner_content->download('xlsx');
        } else {
            return view('admin.banners.banner_does_not_exist');
        }
    }

    /**
     *
     *
     *
     * RESTFUL METHODS
     *
     *
     *
     */

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index(Request $request)
    {

        //if a query exists from Scout get the string
        $query = $request->input('query');

        //check if the query is empty if it isn't use the scout search method before paginating
        $collection = empty($query) ? Banner::orderby('id', 'DESC')->paginate(10) : Banner::search($query)->paginate(10);

        return view('admin.partials.index_page.index', $this->banner_repository->indexElements(compact('collection')));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        $banner_types = BannerType::pluck('title', 'id');
        $companies    = Banner::pluck('company', 'company');

        return view($this->banner_repository->getAdminFolderPath() . '.create', compact('companies', 'banner_types'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(BannerRequest $request)
    {

        Banner::create($request->all());

        return redirect('banner');
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(Banner $banner)
    {

        $banner_types = BannerType::pluck('title', 'id');
        $companies    = Banner::pluck('company', 'company');
        $newsletter   = Newsletter::whereHas('banners', function ($query) use ($banner) {

            $query->where('banner_id', $banner->id);
        })->first();

        $newsletter = $newsletter ?: false;

        return view($this->banner_repository->getAdminFolderPath() . '.edit', compact('banner', 'companies', 'banner_types', 'newsletter'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function update(BannerRequest $request, Banner $banner)
    {

        $banner->update($request->all());

        return redirect('banner');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     * @throws \Exception
     */
    public function destroy(Banner $banner)
    {

        $banner->delete();

        return redirect('banner');
    }

    /**
     *
     *
     *
     * AJAX METHODS
     *
     *
     *
     */

    /**
     * @param BannerRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */

    public function getBannerList(Request $request)
    {
        //banner type
        $banner_type = $request->banner_type;
        $selected = "";

        if ($banner_type == 'marex_board') {
            $selected = $request->banner_id;
        }

        //get a list of banners that meet that size constraint
        $banners = Banner::BannerListByBannerType($banner_type);

        return view('newsletters.partials.developer.banner_drop_down', compact('banners', 'selected'));
    }

    public function generate_reports(Request $request)
    {
        $start_date = $request->start_date;
        $end_date   = $request->end_date;
        $company   = $request->company;
        if (!empty($request->email)) {
            $user = $request->email;
        } else {
            $user = Auth::user();
        }

        $bannerfiles = [];

        if (!empty($company)) {
            $banners = Banner::whereHas('newsletters', function ($query) use ($start_date, $end_date) {

                $query->where('official_date', '>=', $start_date)->where('official_date', '<=', $end_date);
            })->where('company', '=', $company)->get();
        } else {
            $banners = Banner::whereHas('newsletters', function ($query) use ($start_date, $end_date) {

                $query->where('official_date', '>=', $start_date)->where('official_date', '<=', $end_date);
            })->get();
        }

        foreach ($banners as $banner) {
            $file_name = preg_replace("/[^A-Za-z0-9 ]/", '', 'Banner Marketing Report - ' . $banner->company . ' (' . $banner->line_item . ')');
            $bannerfiles [] = $file_name;
        }

        CreateBannerReports::dispatch($start_date, $end_date, $company);

        return $bannerfiles;
    }

    public function download_report($file)
    {
        $filedownload = storage_path('reports/banners/' . $file);

        return \Response::download($filedownload);
    }

    public function send_reports(Request $request)
    {
        $bannerfiles = $request->reportfiles;
        $email_title = $request->email_title;
        $email_body = $request->email_body;


        if (!empty($request->email)) {
            if (strpos($request->email, ', ') !== false) {
                $emails = explode(', ', $request->email);
            } elseif (strpos($request->email, ',') !== false) {
                $emails = explode(",", $request->email);
            } else {
                $emails = [$request->email];
            }
            $count = 0;
            foreach ($emails as $email_user) {
                $user = $email_user;
                $status = 0;
                if (filter_var($user, FILTER_VALIDATE_EMAIL)) {
                    Mail::send('emails.banner_reportn', ['email_body' => $email_body], function ($message) use ($email_title, $user, $bannerfiles) {
                        $message->to($user)->subject($email_title);
                        foreach (explode(',', $bannerfiles) as $filePath) {
                            $message->attach(storage_path('reports/banners/' . $filePath));
                        }
                    });
                    $status = 1;
                }
                $log = BannerLog::create([
                    'email_to' => $user,
                    'sender_name' => Auth::user()->name,
                    'email_status' => $status,
                    'email_title' => $email_title,
                    'report_files' => $bannerfiles,
                ]);
                $count++;
            }
            if ($count == count($emails)) {
                return 1;
            }
        } else {
            $user = Auth::user();

            Mail::send('emails.banner_reportn', ['email_body' => $email_body], function ($message) use ($email_title, $user, $bannerfiles) {
                $message->to($user->email, $user->name)->subject($email_title);
                foreach (explode(',', $bannerfiles) as $filePath) {
                    $message->attach(storage_path('reports/banners/' . $filePath));
                }
            });

            $log =  BannerLog::create([
               'email_to' => $user->email,
               'sender_name' => $user->name,
               'email_status' => 1,
               'email_title' => $email_title,
               'report_files' => $bannerfiles,
            ]);
            return 1;
        }
    }

    public function history_reports()
    {
        $query = BannerLog::select('banner_logs.*');

        $datagrid = (new GridConfig())
        ->setComponents([
                (new THead())
                    ->setComponents([
                        (new ColumnHeadersRow()),
                        (new FiltersRow()),
                        (new OneCellRow())
                            ->setRenderSection(RenderableRegistry::SECTION_END)
                            ->setComponents([
                                (new HtmlTag())->setTagName('a')
                                ->setAttributes([
                                    'href' => '/' . \Request::path(),
                                    'class' => 'btn btn-sm btn-success'
                                ])
                                ->addComponent(new RenderFunc(function () {
                                    return '<i class="glyphicon glyphicon-refresh"></i> Clear';
                                })),
                                (new HtmlTag())
                                    ->setAttributes(['class' => 'pull-right'])
                                    ->addComponent(new ShowingRecords()),
                            ])
                    ]),( new TFoot()),
            ])
            ->setDataProvider(
                new EloquentDataProvider($query)
            )
            ->setName('filter')
            ->setPageSize(150)
            ->setColumns([
                (new FieldConfig('sender_name'))->setLabel('Sender Name')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true),
                (new FieldConfig('email_to'))->setLabel('Email Sent to')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true),
                (new FieldConfig('email_title'))->setLabel('Email Title')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true),
                (new FieldConfig('report_files'))->setLabel('Links to Files')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)
                    ->setCallback(function ($val, ObjectDataRow $row) {
                        $history = $row->getSrc();
                        $links = '';
                        foreach (explode(',', $history->report_files) as $item) {
                            $links .= '<a href="/banner/download/' . $item . '" target="_blank">' . $item . '</a> ';
                        }
                        return $links;
                    }),
                (new FieldConfig('email_status'))->setLabel('Email Status')
                    ->addFilter((new SelectFilterConfig())
                        ->setOptions([
                            '1' => 'sent',
                            '0' => 'not sent'
                        ])
                        ->setSubmittedOnChange(true)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            if ($val == '1') {
                                $provider->getBuilder()->where('email_status', '=', 1);
                            } else {
                                $provider->getBuilder()->where('email_status', '=', 0);
                            }
                        }))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                            $history = $row->getSrc();
                            if ($history->email_status == 1) {
                                return 'sent';
                            } else {
                                return 'not sent';
                            }
                        }),
                (new FieldConfig('created_at'))->setLabel('Email Date')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true),
            ]);

        $grid = new Grid($datagrid);
        $grid = $grid->render();

        return view('admin.banners.history', compact('grid'));
    }
}
