<?php

namespace App\Http\Controllers;

use App\Repositories\EmailMarketingRepository;

use Excel;
use DB;
use Illuminate\Http\Request;
use PHPExcel_RichText;
use PHPExcel_Style_Alignment;
use PHPExcel_Style_Border;
use PHPExcel_Style_Fill;
use PHPExcel_Style_NumberFormat;
use PHPExcel_Worksheet_Drawing;

class EmailMarketingController extends Controller
{
    protected $emailMarketingRepository;

    public function __construct(EmailMarketingRepository $emailMarketingRepository)
    {

        $this->middleware('auth');

        $this->emailMarketingRepository = $emailMarketingRepository;
    }

    public function index(Request $request)
    {

        //get the query data from Request if there is any
        $query = $request->input('query');

        //get the campaigns records as a collection
        $campaign_records = $this->emailMarketingRepository->getCampaignRecords()->sortByDesc('name');

        $collection = empty($query) ? $this->emailMarketingRepository->paginate($campaign_records) : $this->emailMarketingRepository->filteredAndPaginated($query, $campaign_records);

        $table_path = $this->emailMarketingRepository->getTablePath();

        return view('admin.partials.index_page.index', $this->emailMarketingRepository->indexElements(compact('collection', 'table_path')));
    }

    public function report($campaign_uid)
    {

        //general campaign information
        $mw_campaign = DB::connection('email_marketing')->table('mw_campaign')->select(DB::raw('mw_campaign.name as campaign_name, mw_campaign_group.name as campaign_group_name, mw_campaign.send_at'))->leftJoin('mw_campaign_group', 'mw_campaign_group.group_id', '=', 'mw_campaign.group_id')->where('campaign_uid', '=', $campaign_uid)->first();

        return Excel::create($mw_campaign->campaign_group_name . ' - ' . $mw_campaign->campaign_name, function ($excel) use ($campaign_uid, $mw_campaign) {

            /**
             *
             * Campaign Stats Sheet
             *
             */
            $excel->sheet('Campaign Stats', function ($sheet) use ($campaign_uid, $mw_campaign) {

                /**
                 * queries required for excel sheet
                 */

                //total opens
                $mw_campaign_track_open_total = DB::connection('email_marketing')->table('mw_campaign')->select(DB::raw('COUNT(*) total'))->join('mw_campaign_track_open', 'mw_campaign_track_open.campaign_id', '=', 'mw_campaign.campaign_id')->where('campaign_uid', '=', $campaign_uid)->first()->total;

                //total url clicks
                $mw_campaign_url_total = DB::connection('email_marketing')->table('mw_campaign')->select(DB::raw('COUNT(*) total'))->join('mw_campaign_url', 'mw_campaign_url.campaign_id', '=', 'mw_campaign.campaign_id')->join('mw_campaign_track_url', 'mw_campaign_track_url.url_id', '=', 'mw_campaign_url.url_id')->where('campaign_uid', '=', $campaign_uid)->where('mw_campaign_url.destination', '!=', '[UNSUBSCRIBE_URL]')->first()->total;

                //total clicks per url
                $mw_campaign_url_list = DB::connection('email_marketing')->table('mw_campaign')->select(DB::raw('COUNT(*) total, mw_campaign_url.destination'))->join('mw_campaign_url', 'mw_campaign_url.campaign_id', '=', 'mw_campaign.campaign_id')->join('mw_campaign_track_url', 'mw_campaign_track_url.url_id', '=', 'mw_campaign_url.url_id')->where('campaign_uid', '=', $campaign_uid)->where('mw_campaign_url.destination', '!=', '[UNSUBSCRIBE_URL]')->groupBy('mw_campaign_url.url_id')->orderBy('mw_campaign_url.url_id', 'asc')->get();

                /**
                 * STYLE ITEMS
                 */

                //make the entire page have a white background
                $styleArray = [
                    'fill' => [
                        'type'  => PHPExcel_Style_Fill::FILL_SOLID,
                        'color' => ['rgb' => 'FFFFFF'],
                    ],
                ];

                $styleArray_background_grey = [
                    'fill' => [
                        'type'  => PHPExcel_Style_Fill::FILL_SOLID,
                        'color' => ['rgb' => 'F5F5F5'],
                    ],
                ];

                $styleArray_background_blue = [
                    'fill' => [
                        'type'  => PHPExcel_Style_Fill::FILL_SOLID,
                        'color' => ['rgb' => 'ADD8E6'],
                    ],
                ];

                $styleArray_background_orange = [
                    'fill' => [
                        'type'  => PHPExcel_Style_Fill::FILL_SOLID,
                        'color' => ['rgb' => 'FFD394'],
                    ],
                ];

                $styleArray_BoldText = [
                    'font' => [
                        'bold' => true,
                    ],
                ];

                $BStyle_thin = [
                    'borders' => [
                        'outline' => [
                            'style' => PHPExcel_Style_Border::BORDER_THIN,
                        ],
                    ],
                ];

                $BStyle_all = [
                    'borders' => [
                        'allborders' => [
                            'style' => PHPExcel_Style_Border::BORDER_THIN,
                        ],
                    ],
                ];

                $BStyle_bottom_double = [
                    'borders' => [
                        'bottom' => [
                            'style' => PHPExcel_Style_Border::BORDER_DOUBLE,
                        ],
                    ],
                ];

                //apply defautl styling
                $sheet->getDefaultStyle()->applyFromArray($styleArray);

                /**
                 * adding the marex logo to the top left of the sheet
                 */

                $objDrawing = new PHPExcel_Worksheet_Drawing();
                $objDrawing->setPath('media/images/site/logo/marex_logo.png');
                $objDrawing->setCoordinates('A1');
                $objDrawing->setHeight(92);
                $objDrawing->setWorkSheet($sheet);

                /**
                 * adding in the text
                 */

                $sheet->mergeCells('A6:C6')->cell('A6', 'E-blast Campaign Stats')->getStyle('A6')->applyFromArray($styleArray_BoldText)->getFont()->setSize(14);

                //Company Name
                $company_text = new PHPExcel_RichText();
                $company_bold = $company_text->createTextRun('Company: ');
                $company_bold->getFont()->setBold(true);
                $company_text->createText($mw_campaign->campaign_group_name);

                $sheet->mergeCells('A7:E7')->cell('A7', $company_text);

                //Sendout Date
                $sendout_date_text = new PHPExcel_RichText();
                $sendout_date_bold = $sendout_date_text->createTextRun('Sendout Date: ');

                $sendout_date_bold->getFont()->setBold(true);
                $sendout_date_text->createText($mw_campaign->send_at);

                $sheet->mergeCells('A8:E8')->cell('A8', $sendout_date_text);

                //Total Sendout
                $sendout_total_text = new PHPExcel_RichText();
                $sendout_total_bold = $sendout_total_text->createTextRun('Total Sendout: ');
                $sendout_total_bold->getFont()->setBold(true);
                $sendout_total_text->createText('77402');

                $sheet->mergeCells('A9:C9')->cell('A9', $sendout_total_text);

                //Title: Campaign Overview
                $sheet->mergeCells('A11:C11')->cell('A11', 'Campaign Overview')->getStyle('A11')->applyFromArray($styleArray_BoldText)->getFont()->setSize(14);

                //Company Address Section
                $sheet->mergeCells('I1:M1')->cell('I1', '3200 S. Andrews Ave., Suite 100');
                $sheet->mergeCells('I2:M2')->cell('I2', 'Ft. Lauderdale, FL 33316');
                $sheet->mergeCells('I3:M3')->cell('I3', 'Phone: +1(954) 848-9955');
                $sheet->mergeCells('I4:M4')->cell('I4', 'Fax: +1(954) 848-9948');
                $sheet->mergeCells('I5:M5')->cell('I5', 'www.maritime-executive.com');

                $sheet->mergeCells('I7:M7')->cell('I7', date('F j, Y'));

                $sheet->cells('I1:I7', function ($cells) {

                    $cells->setAlignment('right');
                });

                $sheet->cell('L12', 'Total Clicks')->getStyle('C12')->applyFromArray($styleArray_BoldText);
                $sheet->cell('L13', $mw_campaign_url_total);

                $sheet->cell('M12', 'Total Impressions')->getStyle('D12')->applyFromArray($styleArray_BoldText);
                $sheet->cell('M13', $mw_campaign_track_open_total);

                $sheet->cell('A15', 'Link Position');
                $sheet->mergeCells('B15:K15')->cell('B15', 'URL');
                $sheet->cell('L15', 'Clicks');
                $sheet->cell('M15', 'CTR');

                /**
                 * Apply more global styling
                 */

                //links header
                $sheet->getStyle('A15:M15')->applyFromArray($styleArray_BoldText);
                $sheet->getStyle('A15:M15')->applyFromArray($BStyle_thin);
                $sheet->getStyle('A15:M15')->applyFromArray($BStyle_bottom_double);
                $sheet->getStyle('A15:M15')->applyFromArray($styleArray_background_grey);

                //eblast stats information
                $sheet->getStyle("C1:C7")->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_RIGHT);

                //total click/impression numbers
                $sheet->getStyle("L13:M13")->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_LEFT);

                //total clicks/impressions
                $sheet->getStyle('L12:M13')->applyFromArray($styleArray_background_orange);
                $sheet->getStyle('L12:M13')->applyFromArray($BStyle_all);

                //count set to starting row for links
                //set count here so if no links exist it still creates a border around the links area.
                $count = 16;

                /**
                 * add in the list of links, total clicks, and CTR
                 */
                foreach ($mw_campaign_url_list as $key => $url) {
                    $total       = $url->total;
                    $destination = $url->destination;
                    $count       = $key + 16;

                    //Link Position
                    $sheet->cell('A' . $count, 'Link' . ' ' . ($key + 1));

                    //URL
                    $sheet->mergeCells('B' . $count . ':K' . $count)->cell('B' . $count, $destination);

                    //Clicks
                    $sheet->cell('L' . $count, $total);

                    //CTR and set the CTR to show as a percentage
                    $sheet->cell('M' . $count, $total / 77402)->getStyle('M' . $count)->getNumberFormat()->applyFromArray(['code' => PHPExcel_Style_NumberFormat::FORMAT_PERCENTAGE_00]);

                    //set row left justified
                    $sheet->getStyle('A' . $count . ':M' . $count)->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_LEFT);

                    //gives an alternating background color for the links (white, blue...)
                    if ($count % 2) {
                        $sheet->getStyle('A' . $count . ':M' . $count)->applyFromArray($styleArray_background_blue);
                    }
                }

                //add a thin border around the links
                $sheet->getStyle('A16:M' . $count)->applyFromArray($BStyle_thin);
            });

            /**
             *
             * Email Stats Sheet
             *
             */

            $excel->sheet('Email Stats', function ($sheet) use ($campaign_uid) {

                //list of emails that clicked on a url
                $mw_campaign_url_email_list = DB::connection('email_marketing')->table('mw_campaign')->select(DB::raw('mw_list_subscriber.email, mw_campaign_url.destination'))->join('mw_campaign_url', 'mw_campaign_url.campaign_id', '=', 'mw_campaign.campaign_id')->join('mw_campaign_track_url', 'mw_campaign_track_url.url_id', '=', 'mw_campaign_url.url_id')->join('mw_list_subscriber', 'mw_list_subscriber.subscriber_id', '=', 'mw_campaign_track_url.subscriber_id')->where('campaign_uid', '=', $campaign_uid)->where('mw_campaign_url.destination', '!=', '[UNSUBSCRIBE_URL]')->orderBy('mw_campaign_url.url_id', 'asc')->orderBy('mw_list_subscriber.email', 'asc')->get();

                if (!$mw_campaign_url_email_list->isEmpty()) {
                    /**
                     * starter info
                     */
                    $destination    = $mw_campaign_url_email_list[0]->destination;
                    $column         = 'A';
                    $email_row      = 4;
                    $link_count     = 1;
                    $link_count_row = '1';
                    $link_row       = '2';

                    /**
                     * STYLE ITEMS
                     */

                    $styleArray_background_blue = [
                        'fill' => [
                            'type'  => PHPExcel_Style_Fill::FILL_SOLID,
                            'color' => ['rgb' => 'ADD8E6'],
                        ],
                    ];

                    $styleArray_background_grey = [
                        'fill' => [
                            'type'  => PHPExcel_Style_Fill::FILL_SOLID,
                            'color' => ['rgb' => 'D3D3D3'],
                        ],
                    ];

                    //set the first column/row (keeps the logic in the loop down to a minimum)
                    $sheet->cell($column . $link_count_row, 'Link ' . $link_count)->getStyle($column . $link_count_row)->applyFromArray($styleArray_background_grey);
                    $sheet->cell($column . $link_row, $destination)->getStyle($column . $link_row)->applyFromArray($styleArray_background_blue);
                    $sheet->cell($column . $email_row, $mw_campaign_url_email_list[0]->email);

                    //set column width
                    $sheet->setWidth($column, '30');

                    /**
                     * list each link and the emails that clicked on it (unique emails)
                     */
                    foreach ($mw_campaign_url_email_list as $email_click) {
                        //check if its a new link
                        if ($email_click->destination != $destination) {
                            $destination = $email_click->destination;

                            //set the starting row for the emails
                            $email_row = 4;

                            //move to the next column
                            $column++;

                            //move to the next link
                            $link_count++;

                            //set column width
                            $sheet->setWidth($column, '30');

                            //add column text Link number and Link URL
                            $sheet->cell($column . $link_count_row, 'Link ' . $link_count)->getStyle($column . $link_count_row)->applyFromArray($styleArray_background_grey);
                            $sheet->cell($column . $link_row, $destination)->getStyle($column . $link_row)->applyFromArray($styleArray_background_blue);
                        } else {
                            //next row to add an email
                            $email_row++;
                        }

                        $destination = $email_click->destination; //the url that is clicked on
                        $email       = $email_click->email; //the subscribers email that clicked on the link

                        //add in the email
                        $sheet->cell($column . $email_row, $email);
                    }

                    //set link vertical alignment to top and bold
                    $sheet->cell('A2:' . $column . '2', function ($cells) {

                        $cells->setValignment('top');
                        $cells->setFontWeight('bold');
                    });

                    //set link number row to center and font size 16
                    $sheet->cell('A1:' . $column . '1', function ($cells) {

                        $cells->setAlignment('center');
                        $cells->setFontSize('16');
                    });

                    //set all cells to wrap text
                    $sheet->getStyle('A2:' . $column . '2')->getAlignment()->setWrapText(true);
                }
            });
        })->download('xlsx');
    }
}
