<?php

namespace App\Http\Controllers;

use App\Article;
use App\Banner;
use App\Http\Requests\NewsletterRequest;
use App\Mail\SendTestNewsletter;
use App\Newsletter;
use App\Repositories\ArticlesRepository;
use App\Repositories\MailWizzRepository;
use App\Repositories\NewslettersRepository;
use App\User;
use Auth;
use Carbon\Carbon;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Mail;
use Redirect;
use Throwable;

class NewslettersController extends Controller
{
    private $mailWizzRepository, $newslettersRepository;

    public function __construct(NewslettersRepository $newslettersRepository)
    {

        $this->middleware('auth', [
            'except' => [
                'index',
                'show',
                'getNewsletterTemplateBlock',
                'archive_banner_click',
                'archive_article_click',
            ],
        ]);
        $this->middleware('subscriber.cookie', [
            'only' => [
                'index',
                'show',
            ],
        ]);

        $this->newslettersRepository = $newslettersRepository;
        $this->mailWizzRepository    = new MailWizzRepository('newsletter');
    }

    /**
     *
     *
     *
     * CUSTOM ROUTE METHODS
     *
     *
     *
     */

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function admin(Request $request)
    {

        //if a query exists from Scout get the string
        $query = $request->input('query');

        //check if the query is empty if it isn't use the scout search method before paginating
        $collection = empty($query) ? Newsletter::orderBy('id', 'DESC')->paginate(10) : Newsletter::search($query)->orderBy('id', 'DESC')->paginate(10);

        $users = User::pluck('name', 'email');

        return view('admin.partials.index_page.index', $this->newslettersRepository->indexElements(compact('collection', 'users')));
    }

    /**
     * allows the clicks from the banners running on the newsletter before transition to continue counting and be reported on.
     *
     * @param Request $request
     */
    public function archive_banner_click(Request $request)
    {

        $email         = $request->email;
        $banner_id     = $request->banner_id;
        $newsletter_id = $request->newsletter_id;
        $datetime      = Carbon::now()->toDateTimeString();
        $ip            = $_SERVER['REMOTE_ADDR'];

        DB::connection('marex')
          ->table('newsletter_click_banner')
          ->insert(
              [
                  'email'               => $email,
                  'newsletterbanner_id' => $banner_id,
                  'newsletter_id'       => $newsletter_id,
                  'created'             => $datetime,
                  'modified'            => $datetime,
                  'ip'                  => $ip,
              ]
          );

        $banner = DB::connection('marex')->table('newsletter_banner_content')->where('id', $banner_id)->first();

        if (strpos($banner->url, 'http://') === false && strpos($banner->url, 'https://') === false) {
            $banner->url = 'http://' . $banner->url;
        }

        return Redirect::to($banner->url, 301);
    }

    /**
     * allows the clicks from the banners running on the newsletter before transition to continue counting and be reported on.
     *
     * @param Request $request
     */
    public function archive_article_click(Request $request)
    {

        $article_slug = $request->article_slug;

        return Redirect::to($article_slug, 301);
    }

    /**
     *
     *
     *
     * RESTFUL METHODS
     *
     *
     *
     */

    public function index()
    {

        $newsletter = Newsletter::where([['published_at', '!=', '0000-00-00'], ['published_at', '<=', Carbon::now()]])->orderBy('official_date', 'DESC')->first();

        $newsletter_data = [];

        if ($newsletter) {
            $newsletter_data = $this->newslettersRepository->NewsletterTemplateBlock([
                                                                                         'newsletter_id' => $newsletter->id,
                                                                                     ]);
        }

        $newsletter_data['newsletter_list'] = Newsletter::where([['published_at', '!=', '0000-00-00'], ['published_at', '<=', Carbon::now()]])->orderBy('official_date', 'DESC')->get()->pluck('title', 'id');

        return view('site.newsletter', $newsletter_data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Article $article)
    {

        $articles = $article->select(DB::raw('CONCAT(published_at, " - ", title) AS title'), 'id')->published()->limit(1000)->pluck('title', 'id');

        return view($this->newslettersRepository->getAdminFolderPath() . '.create', compact('articles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(NewsletterRequest $request)
    {

        /*
         * NEWSLETTER
         */

        //saving the newsletter and getting the saved newsletter collection
        $newsletter = Newsletter::create($request->all());

        //syncs the articles and banners to the newsletter collection
        $this->newslettersRepository->syncNewsletterData($newsletter, $request);

        return redirect('newsletter/admin');
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show(Newsletter $newsletter)
    {

        $newsletter_data = $this->newslettersRepository->NewsletterTemplateBlock([
                                                                                     'newsletter_id' => $newsletter->id,
                                                                                 ]);

        return view('site.newsletter', $newsletter_data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(Newsletter $newsletter, Article $article)
    {

        $articles = $article->select(DB::raw('CONCAT(published_at, " - ", title) AS title'), 'id')->published()->limit(1000)->pluck('title', 'id');

        return view($this->newslettersRepository->getAdminFolderPath() . '.edit', compact('newsletter', 'articles'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function update(NewsletterRequest $request, Newsletter $newsletter)
    {

        $request_data = $request->all();

        //check if the newsletter already went out, if it did published_at can not be changed
        if (!empty($newsletter->campaign_uid) && $newsletter->published_at < Carbon::now()->toDateTimeString() && $newsletter->published_at != '0000-00-00 00:00:00') {
            unset($request_data['published_at']);
        }

        //update the newsletter with the relevant request data
        $newsletter->update($request_data);

        //sync the newsletter data
        $this->newslettersRepository->syncNewsletterData($newsletter, $request);

        return redirect('newsletter/admin');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Newsletter $newsletter)
    {

        if ($newsletter->campaign_uid != '') {
            $this->mailWizzRepository->deleteCampaign($newsletter->campaign_uid);
        }

        $newsletter->delete();

        return redirect('newsletter/admin');
    }

    /**
     *
     *
     *
     * AJAX METHODS
     *
     *
     *
     */

    /**
     * Grabs the Newsletter Template Block can except
     *
     * @param
     * view - only string value is "dev" will load special code blocks for backend use only
     *
     * @param NewsletterRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function getNewsletterTemplateBlock(NewsletterRequest $request)
    {

        $dev           = isset($request->dev) && Auth::check() ? $request->dev : false;
        $newsletter_id = $request->newsletter_id;

        $newsletter_data = $this->newslettersRepository->NewsletterTemplateBlock([
                                                                                     'dev'           => $dev,
                                                                                     'newsletter_id' => $newsletter_id,
                                                                                 ]);

        return view('newsletters.index', $newsletter_data);
    }

    /**
     * @param NewsletterRequest $request
     * @param ArticlesRepository $articleRepository
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function getNewsletterTemplateArticleBlock(NewsletterRequest $request, ArticlesRepository $articleRepository)
    {

        $view_data = [];

        $article_id = $request->article_id;
        $image_type = $request->image_type;
        $source     = $request->source;
        $type       = $request->type;
        $view       = $request->view;

        Log::info($request->all());

        $image = [
            $image_type => [
                'source' => $source,
            ],
        ];

        if (!empty($article_id)) {
            //get the article by id
            $article = Article::findOrFail($article_id);

            //sync up the image to the article
            $articleRepository->sync_images($article, $image);

            //get the article with the new image associated to it
            $article = Article::findOrFail($article_id);

            //set the article into the data being sent to the view
            $view_data['article'] = $article;
        }

        $view_data['view']        = $view;
        $view_data['add_article'] = true;

        return view('newsletters.partials.articles.' . $type, $view_data);
    }

    /**
     * @param NewsletterRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function getNewsletterTemplateBannerBlock(NewsletterRequest $request)
    {

        $banner_id              = $request->banner_id;
        $banner_type            = $request->banner_type;
        $newsletter_banner_type = $request->newsletter_banner_type;
        $add_banner             = true;

        $banner = $banner_id ? Banner::findOrFail($banner_id) : Banner::where('company', 'dev')->whereHas('bannerType', function ($query) use ($banner_type) {

            $query->where('slug', $banner_type);
        })->first();

        return view('newsletters.partials.banner', compact('banner', 'banner_type', 'add_banner', 'newsletter_banner_type'));
    }

    public function getNewsletterTemplateCorporateBlock(NewsletterRequest $request)
    {

        $official_date = $request->official_date;

        $articles    = $this->newslettersRepository->getCorporateArticles($official_date);
        $add_article = '1';

        return view('newsletters.partials.articles.corp_news', compact('articles', 'add_article'));
    }

    public function sendNewsletterEmailTest(NewsletterRequest $request)
    {

        $emails        = $request->emails;
        $newsletter_id = $request->newsletter_id;

        Mail::to($emails)->send(new SendTestNewsletter($newsletter_id));
    }
}
