<?php

namespace App\Http\Controllers;

use Grids;
use Nayjest\Grids\Components\FiltersRow;
use Nayjest\Grids\Components\Footer;
use Nayjest\Grids\Components\Header;
use Nayjest\Grids\Components\Pager;
use Nayjest\Grids\Components\RenderFunc;
use Nayjest\Grids\DataProvider;
use Nayjest\Grids\EloquentDataProvider;
use Nayjest\Grids\EloquentDataRow;
use Nayjest\Grids\FieldConfig;
use Nayjest\Grids\FilterConfig;
use Nayjest\Grids\SelectFilterConfig;
use Nayjest\Grids\Grid;
use Nayjest\Grids\GridConfig;
use Nayjest\Grids\IdFieldConfig;
use Nayjest\Grids\ObjectDataRow;

use App\ArticleMagazineEdition;
use App\Jobs\PopulateMailingListTable;
use App\Subscriber;
use App\Country;
use App\ArticleMagazineMonth;
use App\Repositories\MailingListRepository;
use App\Repositories\SubscribersRepository;
use App\Http\Requests\SubscriberRequest;
use App\Jobs\CreateMailingListExcelSheet;
use App\Jobs\UpdateSubscriberEmail;
use App\SubscriberRegion;
use Carbon\Carbon;
use DB;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Schema;

class SubscribersController extends Controller
{
    //Repositories
    private $subscriberRepository;
    /**
     * @var MailingListRepository
     */
    private $mailing_list_repository;

    /**
     * SubscribersController constructor.
     *
     * @param SubscribersRepository $subscribersRepository
     */
    public function __construct(SubscribersRepository $subscribersRepository, MailingListRepository $mailing_list_repository)
    {

        $this->middleware('auth', [
            'except' => [
                'set_region_question',
                'subscribe_thank_you',
                'index',
                'send_subscriber_email',
                'store',
                'update',
                'update_subscriber',
                'update_subscriber_email',
                'update_subscriber_thank_you',
                'webHookUnsubscribe',
                'webHookSubscribe',
            ],
        ]);

        $this->subscriberRepository    = $subscribersRepository;
        $this->mailing_list_repository = $mailing_list_repository;
    }

    /**
     *
     *
     *
     * CUSTOM ROUTE METHODS
     *
     *
     *
     */

    public function send_subscriber_email(Request $request)
    {

        $email = $request->email;

        return view('site.send_subscriber_email', compact('email'));
    }

    public function update_subscriber(Request $request)
    {

        $email      = decrypt($request->email);
        $subscriber = Subscriber::whereEmail($email)->firstOrFail();

        $subscriptions = [
            [
                'name'   => 'magazine',
                'title'  => 'Print Magazine',
                'status' => old('magazine', 0),
            ],
            [
                'name'   => 'digital',
                'title'  => 'Digital Magazine',
                'status' => old('digital', 1),
            ],
            [
                'name'   => 'newsletter',
                'title'  => 'Newsletter',
                'status' => old('newsletter', 1),
            ],
        ];

        $hide_hellobar = true;

        return view('site.update_subscriber', $this->subscriberRepository->formElements(compact('subscriber', 'hide_hellobar', 'subscriptions')));
    }

    public function update_subscriber_thank_you()
    {

        return view('site.subscriber_update_thankyou');
    }

    public function subscribe_thank_you()
    {

        return view('site.subscriber_thankyou');
    }

    public function mailing_list(Request $request)
    {
        $filter          = $request->filter;
        $filter_month_id = $request->filter_month_id;
        $filter_year     = $request->filter_year;
        $filters         = $this->subscriberRepository->getFilters();
        $months          = ArticleMagazineMonth::pluck('name', 'id');
        $current_year    = Carbon::now()->year;
        $next_year       = Carbon::now()->addYear()->year;
        $years           = [$current_year => $current_year, $next_year => $next_year];
        $printings       = ['magazine' => 'Print Magazine', 'digital' => 'Digital Magazine', 'newsletter' => 'Newsletter'];

        $subscriber_id = $request->subscriber_id;

        if ($subscriber_id) {
            $query = Subscriber::select('subscribers.*')
                ->where('magazine', '=', 1)
                ->where('subscribers.id', '=', $subscriber_id)
                ->leftJoin('countries', 'countries.id', '=', 'subscribers.country_id');
        } else {
            $query = Subscriber::select('subscribers.*')
                ->where('magazine', '=', 1)
                ->leftJoin('countries', 'countries.id', '=', 'subscribers.country_id');
        }

        $datagrid = (new GridConfig())
            ->setDataProvider(
                new EloquentDataProvider($query)
            )
            ->setName('filter')
            ->setPageSize(150)
            ->setColumns([
                (new FieldConfig())->setCallback(function ($val, ObjectDataRow $row) {
                    $subscriber = $row->getSrc();
                    return '<input type="button" title="' . ($subscriber->magazine == '1' ? 'Unsubscribe' : 'Subscribe') . '" value="' . ($subscriber->magazine == '1' ? 'x' : 'v') . '" '
                    . 'data-subscription="magazine" '
                    . 'data-subscription_value="' . ($subscriber->magazine == '1' ? '' : '1') . '" '
                    . 'data-subscriber_id="' . $subscriber->id . '" class="subscriptions btn ' . ($subscriber->magazine == '1' ? 'btn-danger' : 'btn-success') . ' btn-block">';
                }),
                (new FieldConfig('company'))->setLabel('Company Name')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $span = '<span class="input-group-btn"><a class="btn btn-default" target="_blank" href="https://www.google.com/search?q=' . urlencode($subscriber->company) . '">></a></span>';
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="company" type="text" value="' . $subscriber->company . '">';
                        return '<div class="input-group">' . $input . $span . '</div>';
                    }),
                (new FieldConfig('firstname'))->setLabel('First Name')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="firstname" type="text" value="' . $subscriber->firstname . ' ">';
                        return $input;
                    }),
                (new FieldConfig('lastname'))->setLabel('Last Name')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="lastname" type="text" value="' . $subscriber->lastname . ' ">';
                        return $input;
                    }),
                (new FieldConfig('address_1'))->setLabel('Address 1')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $span = '<span class="input-group-btn"><a class="btn btn-default" target="_blank" href="https://www.google.com/search?q=' . urlencode($subscriber->address_1) . '">></a></span>';
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="address_1" type="text" value="' . $subscriber->address_1 . '">';
                        return '<div class="input-group">' . $input . $span . '</div>';
                    }),
                (new FieldConfig('address_2'))->setLabel('Address 2')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="address_2" type="text" value="' . $subscriber->address_2 . '">';
                        return $input;
                    }),
                (new FieldConfig('city'))->setLabel('City')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="city" type="text" value="' . $subscriber->city . '">';
                        return $input;
                    }),
                (new FieldConfig('postal_code'))->setLabel('Postal Code')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="postal_code" type="text" value="' . $subscriber->postal_code . '">';
                        return $input;
                    }),
                (new FieldConfig('region'))->setLabel('Region')
                    ->addFilter((new FilterConfig())->setOperator(FilterConfig::OPERATOR_LIKE))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                        $subscriber = $row->getSrc();
                        $input = '<input class="subscriber_update form-control" data-subscriber_id="' . $subscriber->id . '" name="region" type="text" value="' . $subscriber->region . '">';
                        return $input;
                    }),
                (new FieldConfig())->setLabel('Country')->setName('name')
                    ->addFilter((new SelectFilterConfig())
                        ->setName('name')
                        ->setOptions([
                            'domestic' => 'Domestic',
                            'foreign' => 'Foreign'
                        ])
                        ->setSubmittedOnChange(true)
                        ->setFilteringFunc(function ($val, EloquentDataProvider $provider) {
                            if ($val == 'domestic') {
                                $provider->getBuilder()->where('country_id', '=', 840);
                            } else {
                                $provider->getBuilder()->where('country_id', '!=', 840);
                            }
                        }))->setSortable(true)->setCallback(function ($val, ObjectDataRow $row) {
                            $subscriber = $row->getSrc();
                            $countries = Country::get();
                            $input = '<select class="subscriber_update set_region form-control" data-subscriber_id="' . $subscriber->id . '" name="name"><option value="">Countries</option>';
                            foreach ($countries as $country) {
                                if ($country->id == $subscriber->country_id) {
                                    $selected = 'selected';
                                } else {
                                    $selected = '';
                                }
                                $input .= '<option value="' . $country->id . '" ' . $selected . '>' . $country->name . '</option>';
                            }
                            $input .= '</select>';
                            return $input;
                        }),
            ]);

        $grid = new Grid($datagrid);
        return view('admin.subscribers.mailing_list.index', compact('grid', 'filter', 'filter_month_id', 'filter_year', 'months', 'filters', 'years', 'printings'));
    }

    public function admin(Request $request)
    {

        //check if the query is empty if it isn't use the scout search method before paginating
        $collection = $this->subscriberRepository->search($request);

        return view('admin.subscribers.index', $this->subscriberRepository->indexElements(compact('collection')));
    }

    public function daily_subscribers_report()
    {

        $daily_subscribers_report = DB::connection('newsletter')->select(DB::raw('SELECT date, COUNT(date) AS total FROM (SELECT DATE_FORMAT(date_added, \'%Y-%m-%d %W\') AS date FROM mw_list_subscriber ORDER BY date_added DESC) AS item GROUP BY date ORDER BY date DESC'));

        return view('admin.subscribers.daily_subscribers_report', compact('daily_subscribers_report'));
    }

    public function audit()
    {

        $mailing_lists = DB::table('information_schema.tables')
                           ->select('table_name AS mailing_list', DB::raw('REPLACE(REPLACE(table_name, \'_\', \' \'), \'mailing-list \', \' \') AS title'))
                           ->where('table_name', 'LIKE', '%mailing_list%')
                           ->orderBy('table_name', 'DESC')
                           ->pluck('title', 'mailing_list');

        return view('admin.subscribers.audit.index', compact('mailing_lists'));
    }

    /**
     *
     *
     *
     * RESTFUL METHODS
     *
     *
     *
     */

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {

        $subscriptions = [
            [
                'name'   => 'magazine',
                'title'  => 'Print Magazine',
                'status' => old('magazine', 0),
            ],
            [
                'name'   => 'digital',
                'title'  => 'Digital Magazine',
                'status' => old('digital', 1),
            ],
            [
                'name'   => 'newsletter',
                'title'  => 'Newsletter',
                'status' => old('newsletter', 1),
            ],
        ];

        return view('site.subscribe', $this->subscriberRepository->formElements(compact('subscriptions')));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        return view($this->subscriberRepository->getAdminFolderPath() . '.create', $this->subscriberRepository->formElements());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(SubscriberRequest $request)
    {

        //create new subscriber
        $subscriber = Subscriber::create($request->all());

        //sync the subscriber to mailwizz servers
        $this->subscriberRepository->syncSubscriptionLists($subscriber);

        //Method is used by the frontend as well so this check prevents it from sending back unnecessary information
        if (Auth::user()) {
            //return to the subscriber index page
            return redirect('subscriber/admin');
        } else { // send back the new subscriber id
            return redirect('/subscribe/thank-you');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show(Subscriber $subscriber)
    {
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(Subscriber $subscriber)
    {

        return view($this->subscriberRepository->getAdminFolderPath() . '.edit', $this->subscriberRepository->formElements(compact('subscriber')));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function update(SubscriberRequest $request, Subscriber $subscriber)
    {

        Log::info('Updating subscriber');

        //update the subscriber
        $subscriber->update($request->all());

        //sync the subscriber to mailwizz servers
        $this->subscriberRepository->syncSubscriptionLists($subscriber);

        //Method is used by the frontend as well so this check prevents it from sending back unnecessary information
        if (Auth::user()) {
            return redirect('subscriber/admin');
        } else {
            return redirect('/subscriber/update-subscriber/thank-you');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Subscriber $subscriber)
    {

        try {
            $subscriber->delete();
        } catch (Exception $e) {
            Log::debug($e);
        }

        return redirect('subscriber/admin');
    }

    /**
     *
     *
     * AJAX METHODS
     *
     *
     */

    /**
     * Queue up creation of mailing list
     */
    public function create_mailing_list(Request $request)
    {

        $article_magazine_month_id = $request->article_magazine_month_id;
        $year                      = $request->year;
        $printing                  = $request->printing;
        $user                      = Auth::user();

        CreateMailingListExcelSheet::dispatch($article_magazine_month_id, $year, $printing, $user);
    }

    /**
     * @param Request $request
     */
    public function update_subscription(Request $request)
    {
        $subscription       = $request->subscription;
        $subscriber_id      = $request->subscriber_id;
        $subscription_value = $request->subscription_value;

        //Subscribe the user
        Subscriber::findOrFail($subscriber_id)->update([$subscription => $subscription_value]);

        $subscriber = Subscriber::findOrFail($subscriber_id);

        $this->subscriberRepository->updateSubscription($subscriber, $subscription);

        return $subscriber->{$subscription};
    }

    /**
     * Unsubscribe via Webhook
     *
     * @param Request $request
     */
    public function webHookUnsubscribe(Request $request)
    {

        Log::info('Unsubscribe Webhook');

        //GET data from WebHook
        $list_uid = $request->data['list']['list_uid'];
        $email    = $request->data['subscriber']['email'];

        $subscription = $this->subscriberRepository->getSubscriptionTypeByListUid($list_uid);

        Log::info($list_uid);
        Log::info($email);
        Log::info($subscription);

        //Unsubscribe the user
        Subscriber::where('email', $email)->update([$subscription => '']);
    }

    /**
     * Subscribe via Webhook
     *
     * @param Request $request
     */
    public function webHookSubscribe(Request $request)
    {
        Log::info('Subscribe Webhook');

        //GET data from WebHook
        $list_uid = $request->data['list']['list_uid'];
        $email    = $request->data['subscriber']['email'];

        $subscription = $this->subscriberRepository->getSubscriptionTypeByListUid($list_uid);

        Log::info($list_uid);
        Log::info($email);

        //Subscribe the user
        Subscriber::where('email', $email)->update([$subscription => '1']);
    }

    /**
     * @param Request $request
     */
    public function update_subscriber_email(Request $request)
    {

        $email = $request->email;

        //create the job to send the email
        dispatch((new UpdateSubscriberEmail($email)));

        return view('site.subscriber_email_sent_thankyou', compact('email'));
    }

    public function mailing_list_filter(Request $request, Subscriber $subscriber)
    {

        $filter            = $request->filter;
        $countries         = Country::orderBy('name')->pluck('name', 'id');
        $regions           = SubscriberRegion::orderBy('name')->pluck('name', 'id');
        $subscribers       = Subscriber::magazine();
        $subscribers       = $this->subscriberRepository->mailing_list_filters($filter, $subscribers);
        $subscribers_count = $subscribers->count();
        $subscribers       = $subscribers->paginate(100)->appends(['filter' => $filter]);

        return view('admin.subscribers.mailing_list.partials.filter_data', compact('subscribers', 'countries', 'regions', 'subscribers_count'));
    }

    public function mailing_list_filters(Request $request)
    {

        $mailing_list = $request->mailing_list;

        $return_data['age_of_source_data']              = $this->mailing_list_repository->age_of_source_data($mailing_list);
        $return_data['business_and_occupation']         = $this->mailing_list_repository->business_and_occupation($mailing_list);
        $return_data['geographic_canada']               = $this->mailing_list_repository->geographic_canada($mailing_list);
        $return_data['geographic_international']        = $this->mailing_list_repository->geographic_international($mailing_list);
        $return_data['geographic_poss_and_other_areas'] = $this->mailing_list_repository->geographic_poss_and_other_areas($mailing_list);
        $return_data['geographic_us_alaska_and_hawaii'] = $this->mailing_list_repository->geographic_us_alaska_and_hawaii($mailing_list);
        $return_data['geographic_us_states']            = $this->mailing_list_repository->geographic_us_states($mailing_list);
        $return_data['geographic_us_military']          = $this->mailing_list_repository->geographic_us_military($mailing_list);
        $return_data['geographic_us_classified']        = $this->mailing_list_repository->geographic_us_classified($mailing_list);
        $return_data['issue_by_issue']                  = $this->mailing_list_repository->issue_by_issue($mailing_list);
        $return_data['mailing_address']                 = $this->mailing_list_repository->mailing_address($mailing_list);

        return view('admin.subscribers.audit.partials.filters', $return_data);
    }

    public function create_mailing_list_table(Request $request)
    {

        $article_magazine_month_id = $request->article_magazine_month_id;
        $year                      = $request->year;
        $printing                  = $request->printing;

        $article_magazine_month = ArticleMagazineMonth::findOrFail($article_magazine_month_id);
        $table_name             = $this->mailing_list_repository->mailing_list_table_name($article_magazine_month, $year, $printing);

        //stop running if table already exists
        if (Schema::hasTable($table_name)) {
            return 'FALSE';
        }

        $edition = ArticleMagazineEdition::where([
                                                     ['year', $year],
                                                     ['article_magazine_month_id', $article_magazine_month_id],
                                                 ])->get();

        if ($edition->isEmpty()) {
            ArticleMagazineEdition::create([
                                               'year'                      => $year,
                                               'article_magazine_month_id' => $article_magazine_month_id,
                                           ]);
        }

        //create the table
        DB::connection('mysql')->statement('CREATE TABLE `' . $table_name . '` LIKE subscribers;');

        //request that the table be populated
        PopulateMailingListTable::dispatch($article_magazine_month, $year, $printing);

        return 'TRUE';
    }

    public function set_region_question($type)
    {

        $regions = SubscriberRegion::orderBy('name')->pluck('name', 'id');

        return view('admin.subscribers.partials.' . $type, compact('regions'));
    }

    public function set_mailing_list_region_question($type, Request $request)
    {

        $subscriber_id = $request->subscriber_id;

        $regions    = SubscriberRegion::orderBy('name')->pluck('name', 'id');
        $subscriber = Subscriber::findOrFail($subscriber_id);

        return view('admin.subscribers.mailing_list.partials.' . $type, compact('regions', 'subscriber'));
    }
}
