<?php

namespace App\Repositories;

use App\Article;
use App\ArticleAuthor;
use App\ArticleImageType;
use App\ArticleSlugRedirect;
use App\ArticleType;
use App\ArticleCategory;
use App\ArticleTitleTag;
use App\AnaliyticsView;
use Auth;
use Carbon\Carbon;
use Cviebrock\EloquentSluggable\Services\SlugService;
use Log;
use Redirect;
use Analytics;
use Spatie\Analytics\Period;

class ArticlesRepository extends Repository
{

    protected $route_path = 'article';

    protected $admin_folder_path = 'admin.articles';

    protected $create_button_title = 'Create Article';

    protected $heading_title = 'Articles';

    protected $table_path = 'admin.partials.index_page.table_path_by_slug';

    protected $newslettersRepository;

    public function __construct()
    {

        $this->newslettersRepository = new NewslettersRepository();
    }

    /**
     * Generates the data for all dropdowns on the subscriber form
     *
     * Subscribers uses type subtype for flexibility all types and there children are gathered and returned
     *
     * @param null $subscriber
     *
     * @return array
     */
    public function formElements($misc_elements = [])
    {

        $form_elements = [];

        $form_elements['article_types']   = ArticleType::pluck('title', 'id');
        $form_elements['article_authors'] = ArticleAuthor::orderBy('firstname')->get()->pluck('full_name', 'id');
        $form_elements['published_at']    = Carbon::now();
        $form_elements['article_categories'] = ArticleCategory::where('article_type_id', 1)->pluck('title', 'id');
        // $form_elements['article_tags'] = ArticleTitleTag::pluck('name', 'id');

        return array_merge($form_elements, $misc_elements);
    }

    public function syncItems($article, $request)
    {

        //Sync Title Tags
        if ($request->article_title_tags) {
            $article->articleTitleTags()->sync($request->article_title_tags);
        }

        //Sync Images
        if ($request->image_list) {
            $this->sync_images($article, $request->image_list);
        }

        //Short URL
        if ($request->short_url) {
            $this->store_slug_redirect($article, $request->short_url);
        }
    }

    /**
     * @param $article
     * @param $image_list
     */
    public function sync_images($article, $image_list)
    {

        //turn image_list into a collection
        $image_list = collect($image_list);

        //get image types that are coming from form
        $image_types = $image_list->keys()->toArray();

        //get the article images that are not being replaced.
        $safe_articleImages = $article->articleImages->whereNotIn('articleImageType.name', $image_types)->pluck('id')->toArray();

        //get the article images that are being replaced
//        $delete_articleImages = $article->articleImages->whereIn('articleImageType.name', $image_types)->pluck('id')->toArray();
//        ArticleImage::destroy($delete_articleImages);

        //take each image in the list, create and associate it to its type then sync it to the article
        $image_id_list = [];

        $image_list->each(function ($image, $type) use ($article, &$image_id_list) {

            //todo-hammy v2 need to update these images to save an md5 hash snapshot to check against to keep duplicates from being created. Save lots of space.
            if (!empty($image['source'])) {
                $image = ArticleImageType::whereName($type)->first()->articleImages()->create($image);

                array_push($image_id_list, $image->id);
            }
        })->values()->toArray();

        //merge the new images with the ones not being replaced
        $image_id_list = array_merge($image_id_list, $safe_articleImages);

        $article->articleImages()->sync($image_id_list);
    }

    public function store_slug_redirect($article, $short_slug)
    {

        $slug_service_check     = SlugService::createSlug(Article::class, 'slug', $short_slug);
        $article_redirect_check = ArticleSlugRedirect::where('slug', $slug_service_check)->get();

        if ($article_redirect_check->isEmpty()) {
            $article->articleSlugRedirect()->create([
                                                        'slug' => $slug_service_check,
                                                    ]);

            return 'true';
        } else {
            return 'false';
        }
    }

    public function show_article($article_slug, $article_type)
    {

        $article = Article::whereSlug($article_slug)->first();
        $preview = false;

        // if article doesn't exist check the redirect table
        if (!$article) {
            $this->check_article_redirect($article_slug);
        } else {
            $articlelink = '/' . $article->articleType->slug . '/' . $article->slug;

            $views = $article->views;

            $oldanalitics = AnaliyticsView::where('created_at', '<', Carbon::now()->subDays(3))->limit(1000)->get();
            foreach ($oldanalitics as $item) {
                $item->forcedelete();
            }

            if (!Auth::check() && !AnaliyticsView::where('ip', '=', \Request::ip())->where('slug', '=', $articlelink)->where('created_at', '>=', Carbon::now()->subDays(3))->first()) {
                $view = AnaliyticsView::create([
                    'slug' => $articlelink,
                    'count' => 1,
                    'ip' => \Request::ip(),
                ]);

                $article->update([
                    'views' => $views + 1,
                ]);
            }

            $pageviews = $article->views;

            //if article is a draft
            if ($article && ($article->published_at > Carbon::now()->toDateTimeString() || $article->published_at === '0000-00-00 00:00:00')) {
                $preview = true;
            }

            // if (!Auth::check() && $article && ($article->published_at > Carbon::now()->toDateTimeString() || $article->published_at === '0000-00-00 00:00:00')) {

            //     return abort(404);
            // }
        }

        // if article type isn't correct redirect to correct type
        if ($article->articleType->slug != $article_type) {
            return Redirect::to($article->link, 301)->send();
        }
		$article->content = str_replace('/media/images/','https://maritime-executive.com//media/images/',$article->content);
        $side_article_content = Article::randomArticles();

        $top_stories = Article::published()
            ->where('top_article', '!=', '1')
            ->whereNotIn('article_type_id', ['2', '7', '9'])
            ->limit(4)
            ->get();
        $more_top_stories = Article::published()
            ->where('top_article', '!=', '1')
            ->whereNotIn('article_type_id', ['2', '7', '9'])
            ->skip(4)
            ->limit(4)
            ->get();
		 $editorials = Article::published()
            ->where('top_article', '!=', '1')
            ->where('article_type_id', ArticleType::EDITORIAL)
            ->limit(5)
            ->get();
		$related_stories = Article::published()
            ->where('top_article', '!=', '1')
            ->whereNotIn('article_type_id', ['2', '7', '9'])
            ->limit(4)
            ->get();
		$categories = ArticleCategory::whereIn('id', ['6', '5', '3'])
            ->limit(8)
			->orderBy('order', 'asc') 
            ->get();
		
        foreach ($categories as $key=>$category) {
            $articles = $category->articles()->published()->limit(5)->get();
			
            $categories[$key]['top_article'] = $articles->shift();
            $categories[$key]['articles'] = $articles;
        }
		$categoriesupper = $categories;
        return \Response::view('site.article', compact('article', 'pageviews', 'side_article_content', 'preview', 'top_stories', 'more_top_stories', 'editorials', 'related_stories', 'categoriesupper'))->header('Cache-Control', 'no-store, no-cache, must-revalidate, post-check=0, pre-check=0');
    }

    public function check_article_redirect($slug)
    {

        $article_slug_redirect = ArticleSlugRedirect::whereSlug($slug)->first();

        if ($article_slug_redirect) {
            Redirect::to($article_slug_redirect->article->link, 301)->send();
        } else {
            abort(404);
        }
    }

    /**
     * @param $request
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function search($request)
    {

        //if a query exists from Scout get the string
        $query             = $request->input('query');
        $article_author_id = $request->article_author_id;
        $article_type_id   = $request->article_type_id;
        $date              = $request->date;

        //searched items to append to paginated search
        $appends = [];

        //check if the query is empty if it isn't use the scout search method before paginating
        $collection = empty($query) ? Article::whereNotNull('published_at') : Article::search($query);

        //search author
        if (!empty($article_author_id)) {
            $collection = $collection->where('article_author_id', '=', $article_author_id);

            $appends['article_author_id'] = $article_author_id;
        }

        //search article type
        if (!empty($article_type_id)) {
            $collection = $collection->where('article_type_id', '=', $article_type_id);

            $appends['article_type_id'] = $article_type_id;
        }

        //search start at date for published_at
        if (!empty($date)) {
            $collection = $collection->where('published_at', 'LIKE', '%' . $date . '%');

            $appends['date'] = $date;
        }

        $collection = $collection->orderBy('published_at', 'DESC')->paginate(10)->appends($appends);

        return $collection;
    }

    /**
     * @return string
     */
    public function getScriptPath(): string
    {

        return $this->script_path;
    }

    /**
     * @return string
     */
    public function getTablePath(): string
    {

        return $this->table_path;
    }

    /**
     * @return string
     */
    public function getAdminFolderPath(): string
    {

        return $this->admin_folder_path;
    }
}
