<?php

namespace App\Repositories;

use Carbon\Carbon;
use Log;
use MailWizzApi\Config;
use MailWizzApi\Base;
use MailWizzApi\Endpoint\Campaigns;
use MailWizzApi\Endpoint\ListFields;
use MailWizzApi\Endpoint\Lists;
use MailWizzApi\Endpoint\ListSubscribers;
use MailWizzApi\Endpoint\TransactionalEmails;

class MailWizzRepository extends Repository
{
    protected $mailwizz_config_location_list
        = [
            'newsletter'      => 'mailwizz_newsletter',
            // 'newsletter'      => 'mailwizz_test',
            'email_marketing' => 'mailwizz_email_marketing',
            'digital'         => 'mailwizz_digital',
        ];

    private $campaign, $subscriber, $list, $email;

    //General Variables
    private $from_name = 'The Maritime Executive Newsletter';       // required

    private $from_email = 'newsletter@maritime-executive.com';      // required

    private $subject = 'Subject Placeholder';                       // required

    private $reply_to = 'newsletter@maritime-executive.com';        // required

    private $send_at = '0000-00-00 00:00:00';                       // required

    private $plain_text = null;                                     // leave empty to auto generate

    //Campaign Variables
    private $campaign_uid = '';

    private $campaign_name = 'Newsletter';                          // required

    private $url_tracking = 'yes';                                  // yes | no

    private $json_feed = 'no';                                      // yes | no

    private $xml_feed = 'no';                                       // yes | no

    private $plain_text_email = 'yes';                              // yes | no

    private $email_stats = null;                                    // a valid email address where we should send the stats after campaign done

    private $campaign_content = '';

    private $inline_css = 'no';                                     // yes | no

    private $auto_plain_text = 'yes';                               // yes | no

    public $list_by_name = '';                                      // required

    //Subscriber Variables
    private $subscriber_uid = '';                                   // required

    private $body = '';                                             // required

    private $list_uid = '';

    private $status = 'draft';

    /**
     * EmailMarketingRepository constructor.
     */
    public function __construct($subscription_type = null)
    {

        $this->setupApiConnections($subscription_type);
    }

    public function setupApiConnections($subscription_type = null)
    {

        if ($subscription_type) {
            //setup the connection to the mailwiz server
            $config = new Config([
                         'apiUrl'     => Config($this->mailwizz_config_location_list[$subscription_type] . '.api_url'),
                         'publicKey'  => Config($this->mailwizz_config_location_list[$subscription_type] . '.public_key'),
                         'privateKey' => Config($this->mailwizz_config_location_list[$subscription_type] . '.private_key'),
                     ]);

            Base::setConfig($config);

            //setup class objects
            $this->campaign    = new Campaigns();
            $this->subscriber  = new ListSubscribers();
            $this->list        = new Lists();
            $this->email       = new TransactionalEmails();
            $this->list_fields = new ListFields();

            //set the list_uid
            $this->list_uid = Config($this->mailwizz_config_location_list[$subscription_type] . '.list_uid');
        }
    }

    /**
     * Campaign Methods
     */
    public function createCampaign($campaign_data)
    {

        $this->setCampaign($campaign_data);

        Log::info($this->send_at);

        if ($this->send_at != '0000-00-00 00:00:00') {
            return $this->readableServerResponse($this->campaign->create($this->getCampaign()));
        } else {
            Log::info('Published date not set');
            return false;
        }
    }

    /**
     *
     */
    public function updateCampaign($campaign_data)
    {

        $this->setCampaign($campaign_data);

        if (!empty($this->campaign_uid)) {
            return $this->readableServerResponse($this->campaign->update($this->campaign_uid, $this->getCampaign()));
        } else {
            return false;
        }
    }

    /**
     * @return \MailWizzApi\Http\Response
     */
    public function deleteCampaign($campaign_uid)
    {

        return $this->campaign->delete($campaign_uid);
    }

    public function getCampaigns()
    {

        return $this->readableServerResponse($this->campaign->getCampaigns(1, 1000));
    }

    public function getSingleCampaign($campaign_unique_id)
    {

        return $this->readableServerResponse($this->campaign->getCampaign($campaign_unique_id));
    }

    /**
     * @return array
     */
    public function getCampaign()
    {

        return [
            'name'       => $this->campaign_name,
            'from_name'  => $this->from_name,
            'from_email' => $this->from_email,
            'subject'    => $this->subject,
            'reply_to'   => $this->reply_to,
            'send_at'    => $this->send_at,
            'list_uid'   => $this->list_uid,
            'status'     => $this->status,
            'options'    => [
                'url_tracking'     => $this->url_tracking,
                'json_feed'        => $this->json_feed,
                'xml_feed'         => $this->xml_feed,
                'plain_text_email' => $this->plain_text_email,
                'email_stats'      => $this->email_stats,
            ],
            'template'   => [
                'content'         => $this->campaign_content,
                'inline_css'      => $this->inline_css,
                'plain_text'      => $this->plain_text,
                'auto_plain_text' => $this->auto_plain_text,
            ],
        ];
    }

    /**
     * @param $data
     */
    public function setCampaign($campaign_data)
    {

        //campaign data
        $this->campaign_uid     = isset($campaign_data['campaign_uid']) ? $campaign_data['campaign_uid'] : $this->campaign_uid;
        $this->listuid          = isset($campaign_data['list_uid']) ? $campaign_data['list_uid'] : $this->list_uid;
        $this->from_name        = isset($campaign_data['from_name']) ? $campaign_data['from_name'] : $this->from_name;
        $this->from_email       = isset($campaign_data['from_email']) ? $campaign_data['from_email'] : $this->from_email;
        $this->subject          = isset($campaign_data['subject']) ? $campaign_data['subject'] : $this->subject;
        $this->campaign_name    = isset($campaign_data['campaign_name']) ? $campaign_data['campaign_name'] : ($campaign_data['official_date'] ?? '') . ' - ' . $this->subject;
        $this->reply_to         = isset($campaign_data['reply_to']) ? $campaign_data['reply_to'] : $this->reply_to;
        $this->send_at          = isset($campaign_data['send_at']) && !empty($campaign_data['send_at']) ? $campaign_data['send_at'] : $this->send_at;
        $this->url_tracking     = isset($campaign_data['url_tracking']) ? $campaign_data['url_tracking'] : $this->url_tracking;
        $this->json_feed        = isset($campaign_data['json_feed']) ? $campaign_data['json_feed'] : $this->json_feed;
        $this->xml_feed         = isset($campaign_data['xml_feed']) ? $campaign_data['xml_feed'] : $this->xml_feed;
        $this->plain_text_email = isset($campaign_data['plain_text_email']) ? $campaign_data['plain_text_email'] : $this->plain_text_email;
        $this->email_stats      = isset($campaign_data['email_stats']) ? $campaign_data['email_stats'] : $this->email_stats;
        $this->campaign_content = isset($campaign_data['campaign_content']) ? '<html><body>' . $campaign_data['campaign_content'] . '</body></html>' : $this->campaign_content;
        $this->inline_css       = isset($campaign_data['inline_css']) ? $campaign_data['inline_css'] : $this->inline_css;
        $this->plain_text       = isset($campaign_data['plain_text']) ? $campaign_data['plain_text'] : $this->plain_text;
        $this->status           = isset($campaign_data['send_at']) && !empty($campaign_data['send_at']) && $campaign_data['send_at'] != '0000-00-00 00:00:00' ? 'pending-sending' : $this->status;
    }

    /**
     *
     * SUBSCRIBER LIST METHODS
     *
     */

    /**
     * @return array
     */
    public function getList()
    {

        return $this->readableServerResponse($this->list->getList($this->list_uid));
    }

    /**
     * @param $name
     */
    public function setListByName($name)
    {

        $this->list_by_name = isset($name) ? $name : '';
    }

    /**
     * @return string
     */
    public function getListByName()
    {

        return $this->list_by_name;
    }


    /**
     *
     * SUBSCRIBER METHODS
     *
     */

    /**
     * @param $subscriber_uid
     *
     * @return array
     */
    public function getSubscriber($subscriber_uid)
    {

        $response = $this->subscriber->getSubscriber($this->list_uid, $subscriber_uid);

        return $this->readableServerResponse($response)['data']['record'];
    }

    /**
     * @param $subscriber_data
     *
     * @return array
     */
    public function createSubscriber($subscriber_data)
    {

        $response = $this->subscriber->create($this->list_uid, $subscriber_data);

        return $this->readableServerResponse($response);
    }

    /**
     * @param $subscriber_data
     *
     * @return array
     */
    public function updateSubscriber($subscriber_data)
    {

        $response = $this->subscriber->update($this->list_uid, $this->subscriber_uid, $subscriber_data);

        return $this->readableServerResponse($response);
    }

    /**
     * @param $subscriber_data
     *
     * @return array
     */
    public function syncSubscriber($subscriber_data)
    {

        $response = $this->subscriber->createUpdate($this->list_uid, $subscriber_data);

        return $this->readableServerResponse($response);
    }

    /**
     * @return \MailWizzApi\Http\Response
     */
    public function unsubscribeByEmail($email)
    {

        $response = $this->subscriber->unsubscribeByEmail($this->list_uid, $email);

        return $this->readableServerResponse($response);
    }

    /**
     * @return array
     */
    public function subscriberList()
    {

        return $this->readableServerResponse($this->subscriber->getSubscribers($this->list_uid, $pageNumber = 1, $perPage = 10));
    }

    /**
     * @return \MailWizzApi\Http\Response
     */
    public function sendEmail($email_data)
    {

        return $this->email->create($email_data);
    }

    /**
     *
     * LIST FIELD METHODS
     *
     */

    /**
     * get a list of fields and arrange into an array that uses field tag as the key
     */
    public function listFields()
    {

        //get the list of fields via the Mailwizz API
        $mailwizz_response = $this->readableServerResponse($this->list_fields->getFields($this->list_uid));

        //the array of rearranged field data to return
        $fields = [];

        //check if the mailwizz reponse didn't error out
        if ($mailwizz_response['status'] == 'success') {
            //loop through the returned records
            foreach ($mailwizz_response['data']['records'] as $record) {
                //set the tag as the key for each record
                $fields[$record['tag']] = $record;
            }
        }

        return $fields;
    }

    /**
     * get a field from the list by name i.e. COUNTRIES
     *
     * @param $field
     */
    public function listField($field)
    {

        //get the list of fields with the tag as the key
        $fields = $this->listFields();

        //check if field exists, otherwise return NULL
        return isset($fields[$field]) ? $fields[$field] : null;
    }

    /**
     * MISC METHODS
     */

    /**
     * the core information status, error, data, etc. are all private variables to the api this retrieves them
     * so they are available separately
     *
     * @param $response
     *
     * @return array
     */
    public function readableServerResponse($response)
    {

        $readable_data = [];

        $readable_tags = [
            'status',
            'error',
            'campaign_uid',
            'data',
        ];

        foreach ($readable_tags as $tag) {
            if ($response->body->itemAt($tag)) {
                $readable_data[$tag] = $response->body->itemAt($tag);
            }
        }

        return $readable_data;
    }

    /**
     * @param string $list_uid
     */
    public function setListUid(string $list_uid)
    {

        $this->listuid = $list_uid;
    }

    /**
     * @param string $apiUrl
     */
    public function setApiUrl(string $apiUrl)
    {

        $this->apiUrl = $apiUrl;
    }

    /**
     * @param string $publicKey
     */
    public function setPublicKey(string $publicKey)
    {

        $this->publicKey = $publicKey;
    }

    /**
     * @param string $privateKey
     */
    public function setPrivateKey(string $privateKey)
    {

        $this->privateKey = $privateKey;
    }

    /**
     * @return mixed|string
     */
    public function getListUid()
    {

        return $this->list_uid;
    }

    /**
     * @return array
     */
    public function getMailwizzConfigLocationList(): array
    {

        return $this->mailwizz_config_location_list;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function getMailWizzConfigListUidList()
    {

        $config_list = collect();

        foreach ($this->mailwizz_config_location_list as $subscription_type => $location) {
            $config_list->push([
                                   'subscription_type' => $subscription_type,
                                   'list_uid'          => Config($location . '.list_uid'),
                               ]);
        }

        return $config_list;
    }
}
