<?php

namespace App\Repositories;

use App\Article;
use App\Banner;
use App\Newsletter;
use App\ArticleNewsletter;
use Auth;
use Carbon\Carbon;
use function foo\func;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mail;
use Throwable;
use View;

class NewslettersRepository extends Repository
{

    /**
     * Route Settings
     */
    protected $route_path = 'newsletter';

    protected $admin_folder_path = 'admin.newsletters';

    protected $folder_path = 'newsletters';

    /**
     * View Settings
     */
    protected $create_button_title = 'Create Newsletter';

    protected $heading_title = 'Newsletters';

    /**
     * @var MailWizzRepository
     */
    private $mailWizzRepository;

    public function __construct()
    {

        $this->mailWizzRepository = new MailWizzRepository('newsletter');
    }

    /**
     * @param $data
     *
     * @return array
     */
    public function NewsletterTemplateBlock($data)
    {

        $dev           = $data['dev'] ?? false;
        $newsletter_id = $data['newsletter_id'];
        $banner        = new Banner();


        // Set current marex_board banners to last selected in create form
        $firstBannerId = \DB::table('banner_newsletter')->where('type', 'marex_board')->where('order', '0')
                                ->orderBy('newsletter_id', 'desc')->first()->banner_id;
        $secondBannerId = \DB::table('banner_newsletter')->where('type', 'marex_board')->where('order', '1')
                                ->orderBy('newsletter_id', 'desc')->first()->banner_id;

        //set the default items for the newsletter;
        //latest header, default top-news and corp-news, and placeholders for all banner slots
		 
        $newsletter_data = [
            'banner_list'  => [
                'header'       => [
                    Newsletter::latest()->first()->banners()->where('type', 'header')->first(),
                ],
                'marex_board'  => [
                    // $banner->bannerBySlug('top-news'),
                    // $banner->bannerBySlug('corp-news'),
                    $banner->bannerById($firstBannerId),
                    $banner->bannerById($secondBannerId),
                ],
                'leader_board' => [
                    $banner->bannerBySlug('dev-728x90'),
                    $banner->bannerBySlug('dev-728x90'),
                    $banner->bannerBySlug('dev-728x90'),
                ],
                'corp_news'    => [
                    $banner->bannerBySlug('dev-185x300'),
                ],
            ],
            'article_list' => [
                'corp_news' => Article::newsletterCorporateNews()->get(),
            ],
            'corp_articles' => [],
        ];

		$newsletter_data['top_storiesnews'] = Article::published()
            ->where('top_article', '!=', '1')
            ->whereNotIn('article_type_id', ['2', '7', '9'])
            ->limit(1)
            ->get();
			
		
        //get the newsletter
        if (!empty($newsletter_id)) {
            $corp_articles = ArticleNewsletter::where('newsletter_id', '=', $newsletter_id)->where('type', '=', 'corp_news')->orderBy('order')->pluck('article_id');

            $newsletter                      = Newsletter::findOrFail($newsletter_id);
            $banner_list                     = $newsletter->banners()->orderBy('order')->get();
            $article_list                    = $newsletter->articles()->where('newsletter', '!=', '1')->orderBy('order')->get();
            $corporate_article_list          = $this->getCorporateArticles($newsletter->official_date);
            $newsletter_data['article_list'] = [];
            $newsletter_data['corp_articles'] = [];

            //Create the Banner List
            foreach ($banner_list as $banner) {
                $newsletter_data['banner_list'][$banner->pivot->type][$banner->pivot->order] = $banner;
            }

            //Create the Article List
            foreach ($article_list as $article) {
				
                if (!isset($newsletter_data['article_list'][$article->pivot->type])) {
                    $newsletter_data['article_list'][$article->pivot->type] = collect();
                }

                $key = $corporate_article_list->where('slug', $article->slug)->keys()->first();
                $corporate_article_list->forget($key);

                $newsletter_data['article_list'][$article->pivot->type][$article->pivot->order] = $article;
            }

            $newsletter_data['article_list']['top_articles'] = collect($article_list->take(4)->all());

            if ($corporate_article_list->isNotEmpty()) {
                $newsletter_data['article_list']['corp_news'] = collect();

                $corporate_article_list->each(function ($item, $key) use ($newsletter_data) {

                    $newsletter_data['article_list']['corp_news']->push($item);
                });
            }

            foreach ($corp_articles as $item) {
                $newsletter_data['corp_articles'] [] = $item;
            }
        }

        //if dev is true then this is for the backend
        //only sent from get_newsletter_template on newsletter.form

        $newsletter_data['add_banner']  = false;
        $newsletter_data['add_article'] = false;
		$newsletter_data['add_ads'] = true;

        if ($dev && Auth::check()) {
            $newsletter_data['add_banner']  = true;
            $newsletter_data['add_article'] = true;
        }

        return $newsletter_data;
    }

    /**
     * syncs the articles and banners to the newsletter collection
     *
     * @param $newsletter
     * @param $request
     */
    public function syncNewsletterData($newsletter, $request)
    {

        /*
         * BANNERS
         */

        //get the list of banners
        $banner_list = $request->banners;

        //translate to the array structure required by sync
        $banners = $this->setNewsletterSyncLists($banner_list);

        //syncing up the banners
        $newsletter->banners()->sync($banners);

        /**
         * ARITCLES
         */

        //get the list of articles
        $article_list = $request->articles;

        if ($article_list) {
            //translate to the array structure required by sync
            $articles = $this->setNewsletterSyncLists($article_list);

            //syncing up the articles
            $newsletter->articles()->sync($articles);

            // if published_at isn't set then you can't use this method. Banners are setup before the published date is entered so it will be blank when it's created
            //
            // sync the corporate news after articles are synced. This is because they are only sorted not entered
            // in the newsletter. They are added from the article section. This allows them to be added without
            // also going to the newsletter section and re-updating the specific newsletter that is set for that published_at date
            if ($newsletter->official_date) {
                $this->corporate_news_article_sync($newsletter->official_date);
            }
        }

        if ($newsletter->active == 1) {
            //send latest version over to mailwizz server
            $this->syncMailWizz($newsletter);
        }
    }

    /**
     * @param $newsletter
     */
    public function syncMailWizz($newsletter)
    {

        if ($newsletter instanceof Model && isset($newsletter->id)) {
            //get the campaign content
            $newsletter_data = $this->NewsletterTemplateBlock([
                                                                  'newsletter_id' => $newsletter->id,
                                                              ]);

            try {
                $campaign_content = view('newsletters.index', $newsletter_data)->render();
            } catch (Throwable $e) {
                Log::error($e);
            }


            if (isset($newsletter->campaign_uid) && !empty($newsletter->campaign_uid)) {
                //update the campaign
                $email_marketing_campaign = $this->mailWizzRepository->updateCampaign([
                                                          'subject'          => $newsletter->subject,
                                                          'send_at'          => $newsletter->published_at,
                                                          'official_date'    => $newsletter->official_date,
                                                          'campaign_uid'     => $newsletter->campaign_uid,
                                                          'campaign_content' => $campaign_content,
                                                      ]);

                if (isset($email_marketing_campaign['error']) && $email_marketing_campaign['error'] == 'Requested campaign does not exist.') {
                    $email_marketing_campaign = $this->mailWizzRepository->createCampaign([
                                                          'subject'          => $newsletter->subject,
                                                          'send_at'          => $newsletter->published_at,
                                                          'official_date'    => $newsletter->official_date,
                                                          'campaign_content' => $campaign_content,
                                                      ]);
                }
            } else {
                $email_marketing_campaign = $this->mailWizzRepository->createCampaign([
                                                          'subject'          => $newsletter->subject,
                                                          'send_at'          => $newsletter->published_at,
                                                          'official_date'    => $newsletter->official_date,
                                                          'campaign_content' => $campaign_content,
                                                      ]);


                if (isset($email_marketing_campaign['campaign_uid'])) {
                    $newsletter->update([
                                            'campaign_uid' => $email_marketing_campaign['campaign_uid'],
                                        ]);
                } else {
                    Log::debug($email_marketing_campaign['status']);
                    Log::debug($email_marketing_campaign['error']);
                }
            }
        }

        return $email_marketing_campaign;
    }

    /**
     * temp placement of method
     *
     * accepts an array of [type][] = id
     * transforms them into an array of [id] => ['type' = 'xxx', 'order' = 'x']
     * id's must be unique coming in
     */
    public function setNewsletterSyncLists($array)
    {

        $list = []; //used to return array of transformed data
        foreach ($array as $type => $ids) {
            foreach ($ids as $order => $id) {
                $list[$id] = [
                    'order' => $order,
                    'type'  => $type,
                ];
            }
        }

        return $list;
    }

    /**
     * This section syncs the corporate article with the newsletter based on published date.
     * Any new articles that are not part of the latest newsletter save/update are prepended to the list
     *
     */
    public function corporate_news_article_sync($official_date = null)
    {

        $official_date = $official_date ?? Carbon::now()->toDateString();
        $newsletter    = Newsletter::where('official_date', $official_date)->first();
        $articles      = $this->getCorporateArticles($official_date);

        if ($newsletter) {
            $newsletter_id = $newsletter->id;

            if ($articles->isEmpty()) {
                return false;
            }

            //add 1 to order so it is prepped for next article
            $order = ($articles->last()->order + 1);

            $article_newsletter_array = [];

            //walk through the new articles
            $articles->each(function ($item) use (&$order, $newsletter_id, &$article_newsletter_array) {

                $article_newsletter_array[$item->article_id ?? $item->id] = [
                    'order'         => $item->order ?? $order,
                    'type'          => 'corp_news',
                    'article_id'    => $item->article_id ?? $item->id,
                    'newsletter_id' => $item->newsletter_id ?? $newsletter_id,
                ];

                $order++;
            });

            $newsletter->articles()->wherePivot('type', 'corp_news')->detach();
            $newsletter->articles()->attach($article_newsletter_array);

            return $newsletter;
        } else {
            return null;
        }
    }

    /**
     * @return string
     */
    public function getAdminFolderPath(): string
    {

        return $this->admin_folder_path;
    }

    public function getCorporateArticles($official_date)
    {

        $newsletter             = Newsletter::where('official_date', $official_date)->first();
        $corporate_article_list = Article::newsletterCorporateNews($official_date)->get()->keyBy('id');

        if ($newsletter) {
            $corporate_article_list_check = $newsletter->articles()->where('newsletter', '1')->orderBy('order')->get()->keyby('id');
            $corporate_article_list_check = $corporate_article_list_check->merge($corporate_article_list->diff($corporate_article_list_check));
        }

        return $corporate_article_list_check ?? $corporate_article_list;
    }
}
