<?php

namespace app\Repositories;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Route;
use Request;

class Repository
{

    protected $admin_folder_path, $folder_path, $route_path, $create_button_title, $heading_title, $table_path, $script_path, $more_articles;

    /**
     * @param array $elements
     *
     * @return array
     */
    public function indexElements($elements = [])
    {

        $default_elements = [
            'admin_folder_path'   => $this->admin_folder_path,
            'folder_path'         => $this->folder_path,
            'route_path'          => $this->route_path,
            'create_button_title' => $this->create_button_title,
            'heading_title'       => $this->heading_title,
            'table_path'          => $this->table_path,
            'script_path'         => $this->script_path,
        ];

        return array_merge($default_elements, $elements);
    }

    /**
     * @param       $items
     * @param int   $perPage
     * @param array $options
     *
     * @return LengthAwarePaginator
     */
    public function paginate($items, $perPage = 10, $options = [])
    {

        //makes sure that $items is a collection before using it
        $items = $items instanceof Collection ? $items : collect($items);

        $current_page       = Request::input('page', 1);
        $offset             = ($current_page * $perPage) - $perPage;
        $current_page_items = $items->slice($offset, $perPage);

        return new LengthAwarePaginator($current_page_items, $items->count(), $perPage, Paginator::resolveCurrentPage(), ['path' => Paginator::resolveCurrentPath()]);
    }

    /**
     * uses a fuzzy search to find any reference to the needle in either the key or value in the haystack
     *
     * @param $search
     * @param $items
     *
     * @return Collection
     */
    public function filter($query, $items)
    {

        //makes sure that $items is a collection before using it
        $items = $items instanceof Collection ? $items : collect($items);

        $filtered_items = $items->filter(function ($value, $key) use ($query) {

            return preg_grep('/(' . $query . ')+/i', $value) ? $value : null;
        });

        return $filtered_items;
    }

    /**
     * filters then paginates the items
     *
     * @param       $search
     * @param       $items
     * @param int   $perPage
     * @param array $options
     *
     * @return LengthAwarePaginator - always returns paginated results
     */
    public function filteredAndPaginated($query, $items, $perPage = 10, $options = [])
    {

        $paginated_collection = $this->paginate($this->filter($query, $items), $perPage, $options);

        //add search as a request item to the paginated links
        $paginated_collection->appends(['query' => $query])->links();

        return $paginated_collection;
    }

    /**
     * @param Request $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function moreArticles()
    {

        $limit    = Request::input('limit');
        $offset   = Request::input('offset');

        return view('site.partials.article.more_articles_content', compact('articles'));
    }
}
