<?php

namespace App\Repositories;

use App\Jobs\UpdateSubscriber;
use App\Subscriber;
use App\SubscriberAgeGroup;
use App\SubscriberEducationLevel;
use App\SubscriberIncomeBracket;
use App\SubscriberIndustry;
use App\SubscriberPosition;
use App\SubscriberPurchaseInput;
use App\SubscriberRegion;
use App\SubscriberSource;
use App\SubscriberSubType;
use App\SubscriberType;
use App\Country;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Webpatser\Countries\Countries;

class SubscribersRepository extends Repository
{
    protected $mailwizz_repository_list = [];

    protected $route_path = 'subscriber';

    protected $admin_folder_path = 'admin.subscribers';

    protected $create_button_title = 'Create Subscriber';

    protected $heading_title = 'Subscribers';

    private $server_whit_list
        = [
            'newsletter'      => 'newsletter',
            'digital'         => 'digital',
            'email_marketing' => 'email_marketing',
        ];

    private $filters
        = [
            'address_1_is_blank'                                  => 'Address 1 is Blank',
            'domestic_only_order_by_postal_code'                  => 'Domestic only Order By Postal Code',
            'domestic_region_is_blank_but_address_1_is_not_blank' => 'Domestic Region is Blank but Address 1 is not blank',
            'format_domestic_zip_code'                            => 'Format Domestic Zip Code',
            'no_mag'                                              => 'No Mag',
            'address_2_is_not_blank'                              => 'Address 2 is not Blank',
            'check_for_(_or_)_in_all_fields'                      => 'Check for ( or ) in all fields',
            'domestic_postal_code_with_alpha_characters'          => 'Domestic Postal Code with Alpha Characters',
            'foreign_order_by_zip_code'                           => 'Foreign ordered by zip code',
            'po_boxes_and_address_2_is_not_empty'                 => 'P.O. Boxes and Address 2 is not Empty',
            'us_region_is_not_a_state_code'                       => 'U.S with Incorrect Region',
            'us_territories'                                      => 'U.S. Territories',
            'postal_code_does_not_contain_alpha_numeric'          => 'Postal Code doesn\'t contain Alphanumerics',
            'any_region_or_city_with_intl'                        => 'Any Region or City with INTL',
            'city_equals_region'                                  => 'City equals Region',
            'country_is_blank'                                    => 'Country is Blank',
            'find_all_special_characters_in_address'              => 'Find all Special Characters in Address',
            'united_arab_emirates_ordered_by_address_1'           => 'United Arab Emirates ordered by Address 1',
            '41_characters_or_more'                               => '41 Characters or More',
        ];

    /**
     * Generates the data for all dropdowns on the subscriber form
     *
     * Subscribers uses type subtype for flexibility all types and there children are gathered and returned
     *
     * @param null $subscriber
     *
     * @return array
     */
    public function formElements($misc_elements = [])
    {

        $form_elements = [];                    //returned array

        $form_elements['subscriber_age_groups']       = SubscriberAgeGroup::pluck('name', 'id');
        $form_elements['subscriber_education_levels'] = SubscriberEducationLevel::pluck('name', 'id');
        $form_elements['subscriber_income_brackets']  = SubscriberIncomeBracket::pluck('name', 'id');
        $form_elements['subscriber_industries']       = SubscriberIndustry::pluck('name', 'id');
        $form_elements['subscriber_positions']        = SubscriberPosition::pluck('name', 'id');
        $form_elements['subscriber_purchase_inputs']  = SubscriberPurchaseInput::pluck('name', 'id');
        $form_elements['subscriber_sources']          = SubscriberSource::pluck('name', 'id');
        $form_elements['countries']                   = Countries::orderBy('name')->pluck('name', 'id')->all();
        $form_elements['regions']                     = SubscriberRegion::orderBy('name')->pluck('name', 'id');
        $form_elements['subscriptions']               = [
            [
                'name'   => 'magazine',
                'title'  => 'Print Magazine',
                'status' => old('magazine', 1),
            ],
            [
                'name'   => 'digital',
                'title'  => 'Digital Magazine',
                'status' => old('digital', 1),
            ],
            [
                'name'   => 'newsletter',
                'title'  => 'Newsletter',
                'status' => old('newsletter', 1),
            ],
            [
                'name'   => 'email_marketing',
                'title'  => 'Email Marketing',
                'status' => old('email_marketing', 1),
            ],
        ];

        /**
         * put the united states at the front of the array
         */
        $united_states = $form_elements['countries'][840];

        unset($form_elements['countries'][840]);

        $form_elements['countries'] = array_prepend($form_elements['countries'], $united_states, 840);

        //return the array of elements
        return array_merge($form_elements, $misc_elements);
    }

    /**
     * syncs all items required by the subscriber
     *
     * @param $subscriber
     * @param $request
     */
    public function queueSyncSubscriptionLists($subscriber)
    {

        //add this subscriber to the que to be subscribed/unsubscribed on the mailwizz servers
        dispatch((new UpdateSubscriber($subscriber)));
    }

    /**
     * syncs the subscribers subscriptions to the mailwizz servers.
     *
     * Uses the name of the subscription with no spaces so Newsletter, EmailMarketing to activate the class so
     * NewsletterSubscriptionRepository, EmailMarketingSubscriptionRepository
     *
     * to add more just setup the subtype/class and set the required variables on the class
     *
     * @param $subscriber
     */
    public function syncSubscriptionLists($subscriber)
    {

        $subscription_list = [
            'email_marketing',
            'magazine',
            'newsletter',
            'digital',
        ];

        foreach ($subscription_list as $subscription) {
            $this->updateSubscription($subscriber, $subscription);
        }
    }

    /**
     * Toggles a subscribers subscription from subscribed to unsubscribed internally and across all MailWizz platforms.
     *
     * @param $email
     * @param $subscription_type
     * @param $subscription_check
     *
     * @return bool
     */
    public function updateSubscription($subscriber, $subscription)
    {

        $subscriber_email   = $subscriber->email;
        $subscription_check = $subscriber->{$subscription} == '1' ?: false;
        $response           = [];

        Log::info('Email: ' . $subscriber->email);

        if (isset($this->server_whit_list[$subscription])) {
            $mailWizzRepository = new MailWizzRepository($subscription);

            if ($subscription_check) {
                $response = $mailWizzRepository->syncSubscriber([
                                                                    'EMAIL'   => $subscriber_email,
                                                                    'details' => [
                                                                        'status' => 'confirmed',
                                                                    ],
                                                                ]);

                if ($response['status'] == 'error' && $response['error'] == 'This email address is blacklisted.') {
                    Log::info('Blacklist triggered turn subscription off');

                    $subscriber->update([$subscription => '0']);
                }

                Log::notice('Subscribe Email for ' . $subscription . ': ' . $subscription);
                Log::debug($response);
            } else {
                $response = $mailWizzRepository->unsubscribeByEmail($subscriber_email);

                Log::notice('Unsubscribe Email for ' . $subscription . ': ' . $subscriber_email);
                Log::debug($response);
            }
        } else {
            Log::notice($subscription . ' Subscription does not have a Mailwizz server');
            Log::notice('Subscription has been ' . ($subscription_check ? 'subscribed' : 'unsubscribed') . '.');
        }
    }

    /**
     * Uses the subscriber email and the list_uid provided by the webhook to find the subscriber and the subscription
     *
     * @param $request
     *
     * @return array - subscriber, subscription
     */
    public function getSubscriptionTypeByListUid($list_uid)
    {

        //Find the correct subscription type to unsubscribe based on list_uid
        $mail_wizz_repository = new MailWizzRepository();
        $mailwizz_config_list = $mail_wizz_repository->getMailWizzConfigListUidList();
        $subscription         = $mailwizz_config_list->where('list_uid', $list_uid)->first();

        return $subscription['subscription_type'];
    }

    /**
     * @return string
     */
    public function getAdminFolderPath(): string
    {

        return $this->admin_folder_path;
    }

    public function search($request)
    {

        $subscriber = new Subscriber();

        $country_id = [];
        $industry_id = [];
        $position_id = [];

        if (!empty($request->country)) {
            $country_id = Country::where('name', 'LIKE', '%' . $request->country . '%')->pluck('id');
        }

        if (!empty($request->industry)) {
            $industry_id = SubscriberIndustry::where('name', 'LIKE', '%' . $request->industry . '%')->pluck('id');
        }

        if (!empty($request->position)) {
            $position_id = SubscriberPosition::where('name', 'LIKE', '%' . $request->position . '%')->pluck('id');
        }

        $subscriber = empty($request->id) ? $subscriber : $subscriber->where('id', 'LIKE', '%' . $request->id . '%');
        $subscriber = empty($request->email) ? $subscriber : $subscriber->where('email', 'LIKE', '%' . $request->email . '%');
        $subscriber = empty($request->firstname) ? $subscriber : $subscriber->where('firstname', 'LIKE', '%' . $request->firstname . '%');
        $subscriber = empty($request->lastname) ? $subscriber : $subscriber->where('lastname', 'LIKE', '%' . $request->lastname . '%');
        $subscriber = empty($request->company) ? $subscriber : $subscriber->where('company', 'LIKE', '%' . $request->company . '%');

        $subscriber = empty($request->country) ? $subscriber : $subscriber->whereIn('country_id', $country_id);
        $subscriber = empty($request->industry) ? $subscriber : $subscriber->whereIn('subscriber_industry_id', $industry_id);
        $subscriber = empty($request->position) ? $subscriber : $subscriber->whereIn('subscriber_position_id', $position_id);
        $subscriber = $subscriber->orderBY('id', 'DESC')
                                 ->paginate(10)
                                 ->appends($request->all());

        return $subscriber;
    }

    public function mailing_list_filters($filter, $subscribers)
    {

        //Where Clause Switch
        switch ($filter) {
            case '41_characters_or_more':
                $subscribers = $subscribers->where(function ($query) {

                    $query->orWhereRaw('CHAR_LENGTH(company) > ?', 40)
                          ->orWhereRaw('CHAR_LENGTH(address_1) > ?', 40)
                          ->orWhereRaw('CHAR_LENGTH(address_2) > ?', 40)
                          ->orWhereRaw('CHAR_LENGTH(city) > ?', 40);
                });
                break;
            case 'address_1_is_blank':
                $subscribers = $subscribers->hasNoAlphaNumeric('address_1');
                break;
            case 'domestic_only_order_by_postal_code':
                $subscribers = $subscribers->countryByName('United States');
                break;
            case 'domestic_region_is_blank_but_address_1_is_not_blank':
                $subscribers = $subscribers->countryByName('United States')
                                           ->hasNoAlpha('region')
                                           ->hasAlphaNumeric('address_1');
                break;
            case 'address_2_is_not_blank':
                $subscribers = $subscribers->hasAlphaNumeric('address_2');
                break;
            case 'format_domestic_zip_code':
                $subscribers = $subscribers->countryByName('United States')
                                           ->whereRaw('CHAR_LENGTH(postal_code) < ?', 5);
                break;
            case 'no_mag':
                $subscribers = $subscribers->where(function ($query) {

                    $query->orLike('firstname', 'no mag')
                          ->orLike('lastname', 'no mag')
                          ->orLike('company', 'no mag')
                          ->orLike('position_title', 'no mag')
                          ->orLike('address_1', 'no mag')
                          ->orLike('address_2', 'no mag')
                          ->orLike('city', 'no mag')
                          ->orLike('region', 'no mag')
                          ->orLike('postal_code', 'no mag')
                          ->orLike('email', 'no mag')
                          ->orLike('phone', 'no mag')
                          ->orLike('note', 'no mag');
                });
                break;
            case 'check_for_(_or_)_in_all_fields':
                $subscribers = $subscribers->where(function ($query) {

                    $query->orLike('address_1', ')')
                          ->orLike('address_2', ')')
                          ->orLike('region', ')')
                          ->orLike('postal_code', ')')
                          ->orLike('address_1', '(')
                          ->orLike('address_2', '(')
                          ->orLike('region', '(')
                          ->orLike('postal_code', '(');
                });
                break;
            case 'domestic_postal_code_with_alpha_characters':
                $subscribers = $subscribers->countryByName('United States')
                                           ->hasAlpha('postal_code');
                break;
            case 'foreign_order_by_zip_code':
                $subscribers = $subscribers->notCountryByName('United States');
                break;
            case 'po_boxes_and_address_2_is_not_empty':
                $subscribers = $subscribers->where(function ($query) {

                    $query->orLike('address_2', 'p o')
                          ->orLike('address_2', 'po')
                          ->orLike('address_2', 'p.')
                          ->orLike('address_2', 'o.')
                          ->orLike('address_2', 'post');
                });
                break;
            case 'us_region_is_not_a_state_code':
                $subscribers = $subscribers->whereNotIn('region', [
                    'AL',
                    'AK',
                    'AZ',
                    'AR',
                    'CA',
                    'CO',
                    'CT',
                    'DE',
                    'FL',
                    'GA',
                    'HI',
                    'ID',
                    'IL',
                    'IN',
                    'IA',
                    'KS',
                    'KY',
                    'LA',
                    'ME',
                    'MD',
                    'MA',
                    'MI',
                    'MN',
                    'MS',
                    'MO',
                    'MT',
                    'NE',
                    'NV',
                    'NH',
                    'NJ',
                    'NM',
                    'NY',
                    'NC',
                    'ND',
                    'OH',
                    'OK',
                    'OR',
                    'PA',
                    'RI',
                    'SC',
                    'SD',
                    'TN',
                    'TX',
                    'UT',
                    'VT',
                    'VA',
                    'WA',
                    'WV',
                    'WI',
                    'WY',
                    'DC',
                ])
                                           ->countryByName('United States');
                break;
            case 'us_territories':
                $subscribers = $subscribers->whereIn('region', [
                    'PR',
                    'GU',
                    'MP',
                    'AS',
                    'UM',
                    'VI',
                ]);
                break;
            case 'postal_code_does_not_contain_alpha_numeric':
                $subscribers = $subscribers->hasNoAlphaNumeric('postal_code');
                break;
            case 'any_region_or_city_with_intl':
                $subscribers = $subscribers->where(function ($query) {
                    $query->orLike('region', 'intl')
                          ->orLike('city', 'intl');
                });
                break;
            case 'city_equals_region':
                $subscribers = $subscribers->whereColumn('region', 'city');
                break;
            case 'country_is_blank':
                $subscribers = $subscribers->whereCountryId('');
                break;
            case 'find_all_special_characters_in_address':
                $subscribers = $subscribers->where(function ($query) {

                    $query->where('address_1', 'REGEXP', '[^a-z0-9 ]+')
                          ->where('address_2', 'REGEXP', '[^a-z0-9 ]+')
                          ->where('city', 'REGEXP', '[^a-z0-9 ]+')
                          ->where('region', 'REGEXP', '[^a-z0-9 ]+');
                });
                break;
            case 'united_arab_emirates_ordered_by_address_1':
                $subscribers = $subscribers->countryByName('United Arab Emirates');
                break;
            default:
                break;
        }

        //OrderBy Clause Switch
        switch ($filter) {
            case 'domestic_only_order_by_postal_code':
            case 'united_arab_emirates_ordered_by_address_1':
            case 'format_domestic_zip_code':
            case 'foreign_order_by_zip_code':
                $subscribers = $subscribers->orderBy('postal_code');
                break;
            case 'address_2_is_not_blank':
                $subscribers = $subscribers->orderBy('country_id')
                                           ->orderBy('region');
                break;
            case 'us_region_is_not_a_state_code':
            case 'us_territories':
                $subscribers = $subscribers->orderBy('region');
                break;
            default:
                $subscribers = $subscribers->orderBy('country_id');
        }

        return $subscribers;
    }

    /**
     * @return array
     */
    public function getFilters()
    {
        return $this->filters;
    }
}
